<?php

namespace Rubicore\Ldap;

use Rubicore\Core\Db_Helper;

final class User_Repository
{
	public function get_by_uid(string $uid) : ?array {
		$uid = preg_replace('/[^A-Za-z0-9\-]/', '', $uid);

		$response = Ldap_Helper::search("(&(cidOrganizationStringMain=*)(cn={$uid})(|(pidcategory=Emp)(pidcategory=Ext)))");

		if (!$response || $response['count'] == 0) {
			return null;
		}

		return $this->parse_ldap_response($response['0']);
	}


	public function list() : array
	{
		$users = Ldap_Helper::search("(&(cidOrganizationStringMain=*)(cn=*)(|(pidcategory=Emp)(pidcategory=Ext)))");
		$all_users = [];

		foreach ($users as $user) {
			$user_object = $this->parse_ldap_response($user);

			if (!$user_object) {
				continue;
			}

			$all_users[] = $user_object;
		}

		return $all_users;
	}


	public function get_old_users(string $last_update) : array
	{
		$user_table = Db_Helper::table('users');
		$user_meta_table = Db_Helper::table('usermeta');

		return Db_Helper::get_rows("
			SELECT * FROM {$user_table} AS u

			INNER JOIN {$user_meta_table} AS um
			ON um.user_id = u.ID
			AND um.meta_key = 'last_ldap_update'
			AND um.meta_value <> %s
			AND um.meta_value <> ''
		", [$last_update]);
	}


	public function get_main_nodes(int $user_id, string $type) : array
	{
		$user_node_table = Db_Helper::table('user_node');
		$category_table = Db_Helper::table('category');
		$category_node_table = Db_Helper::table('category_node');
		$node_table = Db_Helper::table('node');

		return Db_Helper::get_rows("
			SELECT * FROM {$category_table} AS c

			INNER JOIN {$category_node_table} AS cn
			ON cn.category_id = c.id

			INNER JOIN {$node_table} AS n
			ON n.id = cn.node_id

			INNER JOIN {$user_node_table} AS un
			ON un.node_id = n.id
			AND un.user_id = %d
			AND un.main = true

			WHERE c.type = %s
		", [$user_id, $type]);
	}


	public function remove_old_jobs(int $user_id)
	{
		$category_table = Db_Helper::table('category');
		$user_category_repo = new \Rubicore\Core\User_Category_Repository();

		$categories = Db_Helper::get_rows("
			SELECT * FROM {$category_table} AS c
			WHERE c.type = 'pidjob'
		");

		foreach ($categories as $category) {
			$user_category_repo->remove($user_id, $category->id);
		}
	}


	public function update(User_Model $user) : ?int
	{
		$new_user = is_null($user->id);

		$userdata = array(
			'ID' => $user->id,
			'user_login' => $user->uid,
			'user_nicename' => $user->uid,
			'user_email' => $user->email,
			'display_name' => $user->firstName . ' ' . $user->lastName,
			'nickname' => $user->uid,
			'first_name' => $user->firstName,
			'last_name' => $user->lastName,
			'description' => $user->description,
			'meta_input' => [
				'phone1' => $user->mobile,
				'phone2' => $user->phone,
				'disable_login' => false,
				'RUBICORE_LDAP_USER' => true,
				'RUBICORE_LDAP_CIDORGANIZATIONSTRINGMAIN' => $user->groupsMainString,
				'RUBICORE_LDAP_PIDJOBCODE' => $user->pidJobCode
			]
		);

		if ($new_user) {
			$userdata['user_pass'] = wp_hash_password('test');
			$userdata['role'] = 'rubicore-user';
		}

		$user_id = wp_insert_user($userdata);

		if (is_wp_error($user_id)) {
			error_log($user->uid);
			error_log($user_id->get_error_message());
			return null;
		}

		return $user_id;
	}

	public function get_all_users() : array
	{
		$user_table = Db_Helper::table('users');
		$usermeta_table = Db_Helper::table('usermeta');

		$rows = Db_Helper::get_rows("
			SELECT u.ID, u.user_login as uid FROM {$user_table} AS u
			INNER JOIN {$usermeta_table} AS um1
			ON um1.user_id = u.ID
			AND um1.meta_key = 'RUBICORE_LDAP_USER'
		");

		return $rows;
	}

	public function deactivate_user(int $user_id) : void
	{
		$userdata = array(
			'ID' => $user_id,
			'user_login' => \Rubicore\Core\Format_Helper::create_uuid(),
			'user_nicename' => '',
			'user_email' => '',
			'display_name' => 'Inaktiv',
			'first_name' => 'Inaktiv',
			'last_name' => 'Inaktiv',
			'meta_input' => [
				'disable_login' => true,
				'RUBICORE_LDAP_USER_INACTIVE' => true
			]
		);

		$res = wp_insert_user($userdata);

		if (is_wp_error($res)) {
			$error_string = $res->get_error_message();
			print_r($error_string);
		}
	}


	private function parse_ldap_response($user): ?array
	{
		$uid = Ldap_Helper::get_single_value($user, 'uid', null);

		if (!$uid) {
			return null;
		}

		$groups_str = Ldap_Helper::get_multi_value($user, 'cidorganizationstringmv', []);
		$groups_main_str = Ldap_Helper::get_single_value($user, 'cidorganizationstringmain', '');

		$groups = Group_Helper::parse_groups($groups_str);
		$groups_main = Group_Helper::parse_groups($groups_main_str);

		$groups = array_filter($groups, function ($val) use ($groups_main) {
			$added = array_search($val['id'], array_column($groups_main, 'id'), true);

			return !is_numeric($added);
		});

		$jobs = Ldap_Helper::get_single_value($user, 'pidjobcode', '');

		return array(
			'uid' => $uid,
			'groups' => $groups,
			'groups_main' => $groups_main,
			'groups_main_str' => $groups_main_str,
			'first_name' => Ldap_Helper::get_single_value($user, 'givenname', ''),
			'last_name' => Ldap_Helper::get_single_value($user, 'sn', ''),
			'email' => Ldap_Helper::get_single_value($user, 'mail', ''),
			'jobs' => $jobs ? [$jobs] : [],
			'pid_job_code' => $jobs,
			'mobile' => Ldap_Helper::get_single_value($user, 'akmobilenumber', ''),
			'phone' => Ldap_Helper::get_single_value($user, 'akphonenumber', '')
		);
	}
}
