<?php

namespace Rubicore\Ldap;

use \Rubicore\Core\Db_Helper;

final class Group_Repository
{
	private string $node_table;

	function __construct()
	{
		$this->node_table = Db_Helper::table('node');
	}

	public function list() : array
	{
		$users = Ldap_Helper::search("(&(cidOrganizationStringMain=*)(cn=*)(|(pidcategory=Emp)(pidcategory=Ext)))");
		$all_groups = array();

		foreach ($users as $user) {
			$groups_main = Ldap_Helper::get_multi_value($user, 'cidorganizationstringmain', []);
			$groups_mv = Ldap_Helper::get_multi_value($user, 'cidorganizationstringmv', []);
			$groups_arr = Group_Helper::parse_groups(array_merge($groups_main, $groups_mv));

			array_push($all_groups, ...$groups_arr);
		}

		$all_groups = array_reduce($all_groups, function ($acc, $curr) {
			$added = array_search($curr['id'], array_column($acc, 'id'));

			return $added === false ? array(...$acc, $curr) : $acc;
		}, array());


		return $all_groups;
	}


	public function create(array $groups, ?int $category_id) : array
	{
		$groups_created = [];

		foreach ($groups as $group) {
			$external_id = $group->id;
			$parent_external_id = is_null($group->parent) ? "" : $group->parent;
			$name = $group->name;

			if (is_null($external_id)) {
				continue;
			}

			// Check if group exists
			$node = Db_Helper::get_row("SELECT * FROM $this->node_table WHERE value = %s", [$external_id]);

			if (is_null($node)) {
				$node_id = Db_Helper::insert('node', array(
					'value' => $external_id,
					'name' => $name
				), array(
					'%s',
					'%s'
				));


				if ($category_id) {
					Db_Helper::insert('category_node', array(
						'category_id' => $category_id,
						'node_id' => $node_id
					), ['%d', '%d']);
				}

				$groups_created[] = [
					'id' => $external_id,
					'name' => $name,
					'node_id' => $node_id,
					'category_id' => $category_id,
					'parent' => $parent_external_id
				];

				continue;
			}

			// Update name if they differ
			if ($node->name != $name) {
				Db_Helper::update('node', array(
					'name' => $name
				), array(
					'id' => $node->id
				));
			}
		}

		return $groups_created;
	}

	public function update_parents(array $groups): void
	{
		foreach ($groups as $group) {
			$external_id = $group->id;
			$parent_external_id = is_null($group->parent) ? "" : $group->parent;

			if (is_null($external_id)) {
				continue;
			}

			// Check if group exists
			$row = Db_Helper::get_row("SELECT * FROM $this->node_table WHERE value = %s", [$external_id]);

			// Set parent if missing
			if (!is_null($row) && $row->parent == null && $parent_external_id != '') {
				$parent_row = Db_Helper::get_row("SELECT * FROM $this->node_table WHERE value = %s", [$parent_external_id]);

				Db_Helper::update('node', array(
					'parent' => $parent_row->id
				), array(
					'id' => $row->id
				), ['%d']);
			}
		}
	}

	public function get_all_nodes_from_category(int $category_id) : array
	{
		$node_table = Db_Helper::table('node');
		$category_node_table = Db_Helper::table('category_node');

		$rows = Db_Helper::get_rows("
			SELECT
				cn.node_id,
				cn.category_id,
				n.name,
				n.value as id
			FROM {$category_node_table} AS cn
			INNER JOIN {$node_table} AS n
			ON n.id = cn.node_id
			WHERE cn.category_id = %d
		", [$category_id], false, ARRAY_A);

		return $rows;
	}

	public function remove_group(Group_Model $group) : void
	{
		Db_Helper::delete('node', ['id' => $group->node_id], ['%d']);
		Db_Helper::delete('category_node', ['node_id' => $group->node_id], ['%d']);
		Db_Helper::delete('user_node', ['node_id' => $group->node_id], ['%d']);
		Db_Helper::delete('link_node', ['node_id' => $group->node_id], ['%d']);
		Db_Helper::delete('post_node', ['node_id' => $group->node_id], ['%d']);
	}

}
