<?php

namespace Rubicore\Ldap;

final class User_Service
{
	private User_Repository $user_repo;
	private \Rubicore\Core\Node_Repository $node_repo;
	private \Rubicore\Core\Category_Node_Repository $category_node_repo;
	private \Rubicore\Core\User_Category_Repository $user_category_repo;
	private \Rubicore\Core\User_Node_Repository $user_node_repo;
	private array $node_cache = [];

	public function __construct()
	{
		$this->user_repo = new User_Repository();
		$this->node_repo = new \Rubicore\Core\Node_Repository();
		$this->category_node_repo = new \Rubicore\Core\Category_Node_Repository();
		$this->user_category_repo = new \Rubicore\Core\User_Category_Repository();
		$this->user_node_repo = new \Rubicore\Core\User_Node_Repository();
		$this->node_cache = [];
	}

	public function get(string $uid) : ?User_Model
	{
		$user = $this->user_repo->get_by_uid($uid);

		if (is_null($user)) {
			return null;
		}

		$wp_user = get_user_by('login', $uid);
		$wp_user = $wp_user == false ? null : $wp_user;

		return User_Transformer::transform_out($wp_user, $user);
	}

	public function list() : array
	{
		$users = $this->user_repo->list();

		return array_map(function ($user) : User_Model {
			$wp_user = get_user_by('login', $user['uid']);
			$wp_user = $wp_user == false ? null : $wp_user;

			return User_Transformer::transform_out($wp_user, $user);
		}, $users);
	}

	public function sync_single(string $uid) : ?User_Model
	{
		$user = $this->get($uid);

		if (is_null($user)) {
			return null;
		}

		$user_id = $this->user_repo->update($user);
		$is_new_user = is_null($user->id);

		if (is_null($user_id)) {
			return $user;
		}

		if ($is_new_user) {
			$this->update_groups($user_id, $user->groupsMain, true);
			$this->update_groups($user_id, $user->groups, false);
			$this->update_jobs($user_id, $user->jobs);

			return $user;
		}

		if ($this->user_needs_update($user)) {
			$this->remove_main_groups($user_id, $user->groupsMain);
			$this->update_groups($user_id, $user->groupsMain, true);
			$this->update_jobs($user_id, $user->jobs);
		}

		return $user;
	}

	public function sync(array $users) : User_Response_Model
	{
		$added = [];

		foreach ($users as $user) {

			$is_new_user = is_null($user->id);

			if ($is_new_user) {
				$user_id = $this->user_repo->update($user);

				if (is_null($user_id)) {
					continue;
				}

				$this->update_groups($user_id, $user->groupsMain, true);
				$this->update_groups($user_id, $user->groups, false);
				$this->update_jobs($user_id, $user->jobs);

				$added[] = $user;
				continue;
			}

			if ($this->user_needs_update($user)) {
				$user_id = $this->user_repo->update($user);

				$this->remove_main_groups($user_id);
				$this->update_groups($user_id, $user->groupsMain, true);
				$this->update_jobs($user_id, $user->jobs);
			}
		}

		$removed = $this->clear_inactive($users);

		return User_Transformer::transform_response_out($added, $removed);
	}

	public function clear_inactive($users_ldap) : array
	{
		$removed = [];
		$users_internal = $this->user_repo->get_all_users();
		$diffs = array_diff(array_column($users_internal, 'uid'), array_column($users_ldap, 'uid'));

		foreach ($diffs as $key => $value) {
			$removed[] = $users_internal[$key];

			$this->user_repo->deactivate_user($users_internal[$key]->ID);
		}

		return $removed;
	}

	private function user_needs_update($user) : bool
	{
		$meta = get_user_meta($user->id);
		$current_group_main = $meta['RUBICORE_LDAP_CIDORGANIZATIONSTRINGMAIN'][0];
		$current_pidjob = $meta['RUBICORE_LDAP_PIDJOBCODE'][0];
		$update_main_groups = $current_group_main != $user->groupsMainString;
		$update_job = $current_pidjob != $user->pidJobCode;

		if ($user->uid != $user->wp_user_login ||
			$user->email != $user->wp_user_email ||
			$user->firstName != $user->wp_first_name ||
			$user->lastName != $user->wp_last_name ||
			$user->mobile != $meta['phone1'][0] ||
			$user->phone != $meta['phone2'][0] ||
			$update_main_groups ||
			$update_job
		) {
			return true;
		}

		return false;
	}


	private function update_jobs($user_id, $jobs)
	{
		$this->user_repo->remove_old_jobs($user_id);

		foreach ($jobs as $jobs) {
			$node = $this->get_node($jobs);

			if (!$node) {
				continue;
			}

			$category_nodes = $this->category_node_repo->get_by_node_id($node->id);

			foreach ($category_nodes as $cn) {
				$this->user_category_repo->add($user_id, $cn->category_id);
			}
		}
	}

	private function update_groups(int $user_id, array $groups, bool $is_main) : void
	{
		foreach ($groups as $group) {
			$node = $this->get_node($group->id);

			if (!$node) {
				continue;
			}

			if (!$this->user_node_repo->get_by_id($user_id, $node->id)) {
				$this->user_node_repo->add($user_id, $node->id, $is_main);
			}
		}
	}

	private function remove_main_groups(int $user_id) : void
	{
		$nodes = $this->user_repo->get_main_nodes($user_id, 'cidorganizations');

		foreach ($nodes as $node) {
			$this->user_node_repo->remove($user_id, $node->node_id);
		}
	}

	private function get_node(string $key)
	{
		if (array_key_exists($key, $this->node_cache)) {
			return $this->node_cache[$key];
		}

		$node = $this->node_repo->get_by_value($key);
		$this->node_cache[$key] = $node;

		return $node;
	}
}
