<?php

final class RUBICORE_Ldap {

	private static function search($search_query) {

		$options = get_option('ldap');

		$connect = ldap_connect($options['host'], $options['port']) or die("Could not connect to LDAP server.");
		ldap_set_option($connect, LDAP_OPT_PROTOCOL_VERSION, 3);

		if ($options['tsl']) {
			ldap_start_tls($connect);
		}

		$bind = ldap_bind($connect, $options['user'], $options['password']);

		if (!$bind) {
			$errno = ldap_errno($connect);

			if ($errno) {
				RUBICORE_Utils::die($errno);
			}

		}

		$read = ldap_search(
			$connect,
			$options['base'],
			$search_query
		);

		$result = ldap_get_entries($connect, $read);
		ldap_close($connect);

		return $result;
	}



	private static function get_ldap_value($obj, $key, $fallback) {
		if (isset($obj[$key])) {
			if (isset($obj[$key][0])) {
				return $obj[$key][0];
			}

			return $fallback;
		}

		return $fallback;
	}



	private static function get_all_users() {
		$users = self::search("(&(cn=*)(pidcategory=Emp))");
		$all_users = array();

		foreach ($users as $user) {
			$uid = self::get_ldap_value($user, 'uid', null);

			if (!$uid) {
				continue;
			}

			$groups = self::get_ldap_value($user, 'cidorganizationstringmv', array());
			$groups_main = self::get_ldap_value($user, 'cidorganizationstringmain', array());
			$jobs = self::get_ldap_value($user, 'pidjobcode', array());

			$all_users[] = array(
				'uid' => $uid,
				'groups' => self::get_user_groups($groups),
				'groups_main' => self::get_user_groups($groups_main),
				'first_name' => self::get_ldap_value($user, 'givenname', ''),
				'last_name' => self::get_ldap_value($user, 'sn', ''),
				'email' => self::get_ldap_value($user, 'mail', ''),
				'jobs' => is_array($jobs) ? $jobs : [$jobs]
			);
		}

		unset($user);

		return $all_users;
	}


	public static function get_all_groups() {
		$users = self::search("(&(cn=*)(pidcategory=Emp))");
		$all_groups = array();

		foreach ($users as $user) {
			$groups = self::get_ldap_value($user, 'cidorganizationstringmv', array());
			$groups_arr = self::get_user_groups($groups);
			array_push($all_groups, ...$groups_arr);
		}

		unset($user);

		$all_groups = array_reduce($all_groups, function ($acc, $curr) {
			$added = array_search($curr['id'], array_column($acc, 'id'));

			return $added === false ? array(...$acc, $curr) : $acc;
		}, array());

		return $all_groups;
	}



	private static function get_user_groups($cid_organization_strings) {
		if (!is_array($cid_organization_strings)) {
			$cid_organization_strings = [$cid_organization_strings];
		}

		$all_groups = array();

		// Process each incoming strings
		foreach ($cid_organization_strings as $key => $cid_organization_string) {
			if ($key === 'count') {
				continue;
			}
			// Split string to an array and remove dangling empty values
			$groups = array_filter(explode('#', $cid_organization_string), fn($val) => $val != ';');

			// Explode and map the values
			$groups = array_map(fn($val) => explode(';', $val), $groups);

			// Map even more and add parent
			$groups = array_map(function ($val, $index) use ($groups) {
				return array(
					"id" => $val[1],
					"name" => $val[0],
					"parent" => $index == 0 ? null : $groups[$index - 1][1]
				);
			}, $groups, array_keys($groups));

			$groups = array_filter($groups, fn($val) => $val['id'] !== null);

			// Push all groups to the same array
			array_push($all_groups, ...$groups);
		}


		// Remove duplicates
		$all_groups = array_reduce($all_groups, function ($acc, $curr) {
			$added = array_search($curr['id'], array_column($acc, 'id'));

			return $added === false ? array(...$acc, $curr) : $acc;
		}, array());


		return $all_groups;
	}



	private static function update_groups($groups) {
		$response_log = array();
		$table_group = RUBICORE_Db::table('group');

		foreach ($groups as $group) {
			$external_id = $group['id'];
			$parent_external_id = is_null($group['parent']) ? "" : $group['parent'];
			$name = $group['name'];

			if (is_null($external_id)) {
				continue;
			}

			// Check if group exists
			$group_row = RUBICORE_Db::get_row("SELECT * FROM $table_group WHERE external_id = %s", array($external_id));

			if (is_null($group_row)) {
				RUBICORE_Db::insert($table_group, array(
					'external_id' => $external_id,
					'name' => $name,
					'type' => 'group'
				), array(
					'%s',
					'%s'
				));

				$response_log[] = "Inserted new group $name";

				continue;
			}

			// Update name if they differ
			if ($group_row->name != $name) {
				RUBICORE_Db::update($table_group, array(
					'name' => $name
				), array(
					'id' => $group_row->id
				));

				$response_log[] = "Update group name from $group_row->name to $name";
			}
		}

		unset($group);

		foreach ($groups as $group) {
			$external_id = $group['id'];
			$parent_external_id = is_null($group['parent']) ? "" : $group['parent'];

			if (is_null($external_id)) {
				continue;
			}

			// Check if group exists
			$group_row = RUBICORE_Db::get_row("SELECT * FROM $table_group WHERE external_id = %s", array($external_id));

			// Set parent if missing
			if (!is_null($group_row) && $group_row->parent == null && $parent_external_id != '') {
				$parent_row = RUBICORE_Db::get_row("SELECT * FROM $table_group WHERE external_id = %s", [$parent_external_id]);

				RUBICORE_Db::update('group', array(
					'parent' => $parent_row->id
				), array(
					'id' => $group_row->id
				));
			}
		}

		unset($group);

		return $response_log;
	}



	public static function update_user($user) {
		$user_row = get_user_by('login', $user['uid']);

		$userdata = array(
			'user_login' => $user['uid'],
			'user_nicename' => $user['uid'],
			'user_email' => $user['email'],
			'display_name' => $user['first_name'] . ' ' . $user['last_name'],
			'nickname' => $user['uid'],
			'first_name' => $user['first_name'],
			'last_name' => $user['last_name']
		);

		if (!empty($user_row)) {
			$userdata['ID'] = $user_row->ID;
		} else {
			$userdata['user_pass'] = wp_hash_password('test');
		}

		$user_id = wp_insert_user($userdata);

		return $user_id;
	}



	private static function update_user_groups($user_id, $user_data) {
		$user_group_table = RUBICORE_Db::table('user_group');
		$group_table = RUBICORE_Db::table('group');

		foreach ($user_data['groups_main'] as $group) {
			$group_row = RUBICORE_Db::get_row("SELECT * FROM $group_table WHERE external_id = %s", [$group['id']]);

			if (is_null($group_row)) {
				print_r($group);
				continue;
			}

			$row = RUBICORE_Db::get_row(
				"SELECT * FROM $user_group_table
				WHERE user_id = %d
				AND group_id = %d"
			, array($user_id, $group_row->id));

			if (is_null($row)) {
				RUBICORE_Db::insert($user_group_table, array(
					'user_id' => $user_id,
					'group_id' => $group_row->id,
					'main' => 1
				), array(
					'%d',
					'%d',
					'%d'
				));
			}
		}

		unset($group);

		foreach ($user_data['groups'] as $group) {
			$group_row = RUBICORE_Db::get_row("SELECT * FROM $group_table WHERE external_id = %s", [$group['id']]);

			$row = RUBICORE_Db::get_row(
				"SELECT * FROM $user_group_table
				WHERE user_id = %d
				AND group_id = %d"
			, array($user_id, $group_row->id));

			if (is_null($row)) {
				RUBICORE_Db::insert($user_group_table, array(
					'user_id' => $user_id,
					'group_id' => $group_row->id,
					'main' => 0
				), array(
					'%d',
					'%d',
					'%d'
				));
			}
		}

		unset($group);
	}



	private static function update_user_job($user_id, $user_data) {
		$ret = array();
		$user_job_table = RUBICORE_Db::table('user_job');
		$job_table = RUBICORE_Db::table('job');

		$current_jobs = RUBICORE_Db::get_rows(
				"SELECT * FROM $user_job_table
				WHERE user_id = %d"
			, array($user_id));

		$current_jobs = array_map(fn($val) => $val->job_id, $current_jobs);
		$missing_in_db = array();

		foreach ($user_data['jobs'] as $job) {
			$job_row = RUBICORE_Db::get_row("SELECT * FROM $job_table WHERE external_id = %s", [$job]);

			if (!$job_row) {
				array_push($missing_in_db, "$job job not found in Db.");
				continue;
			}

			if (!in_array($job_row->id, $current_jobs)) {
				RUBICORE_Db::insert($user_job_table, array(
					'user_id' => $user_id,
					'job_id' => $job_row->id
				), array(
					'%d',
					'%d'
				));

				array_push($ret, "Add job $job for user $user_id");
			}

			$current_jobs = array_filter($current_jobs, fn($val) => $val !== $job_row->id);
		}

		unset($job);

		foreach ($current_jobs as $job) {
			RUBICORE_Db::delete($user_job_table, array('user_id' => $user_id, 'job_id' => $job), ['%d', '%d']);
			array_push($ret, "Delete job $job for user $user_id");
		}

		unset($job);

		$missing_in_db = array_unique($missing_in_db);

		return [...$ret, ...$missing_in_db];
	}


	public static function register_routes(): void {

		RUBICORE_Api::add_route('rubicore/v1', 'ldap/sync/groups', ['POST'], function (WP_REST_Request $req) {
			$all_groups = self::get_all_groups();


			return self::update_groups($all_groups);
		});

		RUBICORE_Api::add_route('rubicore/v1', 'ldap/sync/users', ['POST'], function (WP_REST_Request $req) {
			$users = self::get_all_users();
			$ret = array();

			foreach ($users as $user) {
				$user_id = self::update_user($user);

				if (!is_numeric($user_id)) {
					$uid = $user['uid'];
					array_push($ret, "Something went wrong updating user $uid");

					continue;
				}

				$job_messages = self::update_user_job($user_id, $user);
				$ret = [...$ret, ...$job_messages];

				self::update_user_groups($user_id, $user);
			}

			return $ret;
		});
	}

}
