<?php

namespace Rubicore\Core;

final class Post_Repository {
	private string $post_table;
	private string $post_category_table;
	private string $post_node_table;
	private string $post_user_table;
	private string $post_editor_table;
	private string $course_user_table;
	private string $postmeta;

	function __construct () {
		$this->post_table = Db_Helper::table('posts');
		$this->post_category_table = Db_Helper::table('post_category');
		$this->post_node_table = Db_Helper::table('post_node');
		$this->post_user_table = Db_Helper::table('post_user');
		$this->post_editor_table = Db_Helper::table('post_editor');
		$this->postmeta = Db_Helper::table('postmeta');
		$this->course_user_table = Db_Helper::table('course_user');
	}

	public function get_by_slug($input, ?string $type) : ?object {
		$query = is_numeric($input) ?
			Db_Helper::prepare("SELECT * FROM $this->post_table AS l WHERE l.id = %d", [$input]) :
			Db_Helper::prepare("SELECT * FROM $this->post_table AS l WHERE l.post_name = %s", [$input]);

		$post = $type ?
			Db_Helper::get_row("{$query} AND l.post_type = %s", [$type]) :
			Db_Helper::get_row($query);

		if (!$post) {
			return null;
		}

		$meta = get_post_meta($post->ID);
		$post->meta = [];

		foreach ($meta as $key => $value) {
			$post->meta[$key] = $value[0];
		}

		$post->media_id = isset($post->meta['mediaId']) ? intval($post->meta['mediaId']) : null;
		$post->node_ids = $this->get_node_ids($post->ID);
		$post->category_ids = $this->get_category_ids($post->ID);
		$post->user_ids = $this->get_user_ids($post->ID);
		$post->editor_ids = $this->get_editor_ids($post->ID);

		return $post;
	}


	public function get(array $params, array $post_types): array
	{
		$q = new Query_Builder_Helper($params, Post_Query::$fields);

		$category_ids = $params['categoryId'] ?? null;
		$node_ids = $params['nodeId'] ?? null;
		$user_ids = $params['userId'] ?? null;
		$course_user_ids = $params['courseAttendeeId'] ?? null;
		$status = $params['status'] ?? ['publish'];

		$category_ids_imp = is_array($category_ids) ? implode(',', $category_ids) : '0';
		$node_ids_imp = is_array($node_ids) ? implode(',', $node_ids) : '0';
		$user_ids_imp = is_array($user_ids) ? implode(',', $user_ids) : '0';
		$course_user_ids_imp = is_array($course_user_ids) ? implode(',', $course_user_ids) : '0';

		$post_types = array_map(fn($val) => Db_Helper::prepare('%s', [$val]), $post_types);
		$post_types = implode(',', $post_types ?: ['']);

		$statuses = array_map(fn ($val) => Db_Helper::prepare('%s', [$val]), $status);
		$statuses = implode(',', $statuses);

		$where = "WHERE p.post_type IN ({$post_types}) AND p.post_status IN ({$statuses})";
		$where = Query_Helper::add_where($q->filter_query, $where, "AND", $q->filter_query);
		$where = Query_Helper::add_where($q->search_query, $where, "AND", $q->search_query);

		$category_node_where = ($category_ids || $node_ids) ? "
			AND (category_id IN ({$category_ids_imp}) OR category_id IS NULL)
			AND (node_id IN ({$node_ids_imp}) OR node_id IS NULL)
		" : "";

		$category_node_where = $course_user_ids ? "
			$category_node_where
			AND (c.user_id IN ({$course_user_ids_imp}))
		" : $category_node_where;

		$filter_where = $where;

		if ($user_ids) {
			$where = str_replace('WHERE', '', $where);
			$filter_where = "
				WHERE (
					{$where}
					AND u.user_id IN ({$user_ids_imp})
				) OR (
					{$where}
					{$category_node_where}
				)
			";
		} else {
			$where = str_replace('WHERE', '', $where);
			$filter_where = "
				WHERE (
					{$where}
					{$category_node_where}
				)
			";
		}

		$where = $filter_where;

		$joins = "
			LEFT JOIN {$this->post_category_table} AS pc
			ON pc.post_id = p.ID
			LEFT JOIN {$this->post_node_table} AS pn
			ON pn.post_id = p.ID
			LEFT JOIN {$this->post_user_table} AS u
			ON u.post_id = p.ID
			LEFT JOIN {$this->course_user_table} AS c
			ON c.post_id = p.ID
		";

		$count_query = "SELECT COUNT(DISTINCT p.ID) AS num_rows FROM $this->post_table AS p $joins $where";

		$total_count = Db_Helper::get_row($count_query)->num_rows;
		Response_Helper::set_pagination_total_header($total_count);

		$posts = Db_Helper::get_rows("SELECT DISTINCT p.* FROM $this->post_table AS p $joins $where $q->order_by_query $q->limit_query");

		foreach ($posts as $post) {
			$meta = get_post_meta($post->ID);
			$post->meta = [];

			foreach ($meta as $key => $value) {
				$post->meta[$key] = $value[0];
			}

			$post->media_id = isset($post->meta['mediaId']) ? intval($post->meta['mediaId']) : null;
			$post->node_ids = $this->get_node_ids($post->ID);
			$post->category_ids = $this->get_category_ids($post->ID);
			$post->user_ids = $this->get_user_ids($post->ID);
			$post->editor_ids = $this->get_editor_ids($post->ID);
		}

		return $posts;
	}

	public function search(string $query, ?array $types): array
	{
		$query_lower = strtolower($query);
		$types_selected = false;
		$partial_match_modifier = 0.5;
		$weights = [
			'title' => 100,
			'excerpt' => 50,
			'content' => 25,
			'type' => [
				'page' => 10,
				'news' => 5
			],
		];

		if(is_array($types) && count($types) > 0){
			$post_types = array_map(fn ($val) => Db_Helper::prepare('%s', [$val]), $types);
			$post_types = implode(',', $post_types ?: ['']);
			$types_selected = true;
		}

		if($types_selected){
			$posts = Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			WHERE p.post_type IN ({$post_types})
			AND p.post_status = 'publish'
			AND (
				p.post_title LIKE '%%%s%%'
				OR p.post_excerpt LIKE '%%%s%%'
				OR p.post_content LIKE '%%%s%%'
			)
		", [$query_lower, $query_lower, $query_lower]);
		} else{
			$posts = Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			WHERE p.post_status = 'publish'
			AND (
				p.post_title LIKE '%%%s%%'
				OR p.post_excerpt LIKE '%%%s%%'
				OR p.post_content LIKE '%%%s%%'
			)
		", [$query_lower, $query_lower, $query_lower]);
		}

		foreach ($posts as $post) {
			$post->weight = 0;

			$title_lower = strtolower($post->post_title);
			$excerpt_lower = strtolower($post->post_excerpt);
			$content_lower = strtolower($post->post_content);

			$matchfound = false;
			$title_word_array = explode(' ', trim($title_lower, "?\!\,\."));

			foreach ($title_word_array as $word){
				if($word == $query_lower) {
					$post->weight = $post->weight + $weights['title'];
					$matchfound = true;
					break;
				}
			}

			if (!$matchfound && str_contains($title_lower, $query_lower)) {
				$post->weight = $post->weight + ($weights['title'] * $partial_match_modifier);
			}

			$matchfound = false;
			$excerpt_word_array = explode(' ', trim($excerpt_lower, "?\!\,\."));

			foreach ($excerpt_word_array as $word) {
				if ($word == $query_lower) {
					$post->weight = $post->weight + $weights['excerpt'];
					$matchfound = true;
					break;
				}
			}

			if (!$matchfound && str_contains($excerpt_lower, $query_lower)) {
				$post->weight = $post->weight + ($weights['excerpt'] * $partial_match_modifier);
			}


			if (str_contains($content_lower, $query_lower. " ") ||
				str_contains($content_lower, " ". $query_lower. " ")) {
				$post->weight = $post->weight + $weights['content'];
			}

			else if (str_contains($content_lower, " ". $query_lower. "?") ||
				str_contains($content_lower, " ". $query_lower. "!") ||
				str_contains($content_lower, " ". $query_lower. ".") ||
				str_contains($content_lower, " ". $query_lower. ",")) {
				$post->weight = $post->weight + $weights['content'];
			}

			else if (str_contains($content_lower, $query_lower)) {
				$post->weight = $post->weight + ($weights['content'] * $partial_match_modifier);
			}

			switch ($post->post_type) {
				case 'page':
					$post->weight = $post->weight + $weights['type']['page'];
					break;

				case 'news':
					$post->weight = $post->weight + $weights['type']['news'];
					break;

				default:
					$post->weight = $post->weight + 0;
					break;
			}

			$meta = get_post_meta($post->ID);
			$post->meta = [];

			foreach ($meta as $key => $value) {
				$post->meta[$key] = $value[0];
			}

			$post->media_id = isset($post->meta['mediaId']) ? intval($post->meta['mediaId']) : null;
			$post->node_ids = $this->get_node_ids($post->ID);
			$post->category_ids = $this->get_category_ids($post->ID);
			$post->user_ids = $this->get_user_ids($post->ID);
			$post->editor_ids = $this->get_editor_ids($post->ID);
		}

		uasort($posts, function ($a, $b) {
			if ($a->weight == $b->weight) {
				if ($a->post_modified == $b->post_modified) {
					return 0;
				}

				return ($a->post_modified > $b->post_modified) ? -1 : 1;
			}

			return ($a->weight > $b->weight) ? -1 : 1;
		});

		return $posts;
	}

	public function get_by_user_id(array $params, int $user_id): array
	{
		$q = new Query_Builder_Helper($params, Post_Query::$fields);

		$where = "WHERE p.post_type <> 'media'";

		$where = Query_Helper::add_where($q->filter_query, $where, "AND", $q->filter_query);
		$where = Query_Helper::add_where($q->search_query, $where, "AND", $q->search_query);

		$where = str_replace('WHERE', '', $where);
		$where = Db_Helper::prepare("
				WHERE (
					{$where}
					AND post_author = %d
				) OR (
					{$where}
					AND (
						SELECT COUNT(*) FROM {$this->post_editor_table} AS pe
					 	WHERE pe.post_id = p.ID AND pe.user_id = %d
						AND pe.type = 'editor'
					) = 1
				)
			", [$user_id, $user_id]);

		$count_query = "SELECT COUNT(DISTINCT p.ID) AS num_rows FROM $this->post_table AS p $where";
		$total_count = Db_Helper::get_row($count_query)->num_rows;
		Response_Helper::set_pagination_total_header($total_count);

		$posts = Db_Helper::get_rows("SELECT DISTINCT p.* FROM $this->post_table AS p $where $q->order_by_query $q->limit_query");

		foreach ($posts as $post) {
			$meta = get_post_meta($post->ID);
			$post->meta = [];

			foreach ($meta as $key => $value) {
				$post->meta[$key] = $value[0];
			}

			$post->media_id = isset($post->meta['mediaId']) ? intval($post->meta['mediaId']) : null;
			$post->node_ids = $this->get_node_ids($post->ID);
			$post->category_ids = $this->get_category_ids($post->ID);
			$post->user_ids = $this->get_user_ids($post->ID);
			$post->editor_ids = $this->get_editor_ids($post->ID);
		}

		return $posts;
	}

	public function create(object $body) : int {
		$entity = Post_Transformer::transform_in($body);

		$post_id = wp_insert_post($entity, false);

		if (!$post_id) {
			Response_Helper::die('Something went wrong creating the post');
		}

		Db_Helper::update('posts', ['post_content' => $entity->post_content], ['ID' => $post_id], ['%s']);

		$this->set_post_meta($post_id, $body);

		return $post_id;
	}

	private function set_post_meta(int $post_id, object $body)
	{
		$meta_fields = [
			'blurbs',
			'info',
			'contact',
			'unpublishedDate',
			'authoredDate',
			'pseudoAuthor',
			'mediaId'
		];

		foreach ($meta_fields as $field) {
			update_post_meta($post_id, $field, property_exists($body, $field) ? $body->{$field} : null);
		}

		if (property_exists($body, 'course') && !is_null($body->course)) {
			foreach ($body->course as $key => $value) {
				update_post_meta($post_id, "course_{$key}", $value);
			}
		}
	}

	public function update(int $post_id, object $body) : void
	{
		$entity = Post_Transformer::transform_in($body);

		$entity->ID = $post_id;
		wp_insert_post($entity, false);

		$slug_input = $entity->post_name ? $entity->post_name : sanitize_title($entity->post_title, $post_id);
		$slug = wp_unique_post_slug($slug_input, $post_id, $entity->post_status, $entity->post_type, $entity->post_parent);

		Db_Helper::update('posts', [
			'post_content' => $entity->post_content,
			'post_name' => $slug
		], ['ID' => $post_id], ['%s', '%s']);

		$this->set_post_meta($post_id, $body);
	}

	public function update_sort(array $posts) : void
	{
		foreach ($posts as $post) {
			Db_Helper::update('posts', array('menu_order' => $post->menu_order), array('id' => $post->ID));
		}
	}

	public function delete(int $id): void
	{
		Db_Helper::delete('posts', array('ID' => $id), ['%d']);
		Db_Helper::delete('post_category', array('post_id' => $id), ['%d']);
		Db_Helper::delete('post_node', array('post_id' => $id), ['%d']);
		Db_Helper::delete('post_user', array('post_id' => $id), ['%d']);
		Db_Helper::delete('post_editor', array('post_id' => $id), ['%d']);
		Db_Helper::delete('postmeta', array('post_id' => $id), ['%d']);
		Db_Helper::delete('course_user', array('post_id' => $id), ['%d']);
	}

	public function get_children(int $post_id) : array
	{
		return Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			WHERE p.post_parent = %d
			AND p.post_type <> 'revision'", [$post_id]);
	}

	public function get_by_media_id(int $media_id) : array
	{
		return Db_Helper::get_rows("
			SELECT * FROM $this->post_table
			INNER JOIN $this->postmeta AS pm
			ON pm.meta_key = 'mediaId'
			AND pm.meta_value = %d", [$media_id]);
	}

	public function get_sort_list(int $parent_id, int $sort, string $type, ?int $post_id) : array
	{
		$where = $post_id ? Db_Helper::prepare("AND p.id != %d", [$post_id]) : "";

		return Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			WHERE p.post_parent = %d
			AND p.menu_order > %d
			AND p.post_type = %s
			{$where}
			ORDER BY p.menu_order ASC
		", [$parent_id, $sort, $type]);
	}

	public function publish_posts() : void {
		$posts = Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			WHERE p.post_status = 'future'
			AND p.post_date < %s
		", [Date_Helper::gmt_now(2)]);

		foreach ($posts as $post) {
			Db_Helper::update('posts', [
				'post_status' => 'publish'
			], [
				'ID' => $post->ID
			], ['%s']);
		}
	}

	public function unpublish_posts() : ?array
	{
		$posts = Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			INNER JOIN {$this->postmeta} AS pm
			ON pm.post_id = p.ID
			AND pm.meta_key = 'unpublishedDate'
			AND pm.meta_value < %s
			WHERE p.post_status = 'publish'
		", [Date_Helper::gmt_now(2)]);

		foreach ($posts as $post) {
			Db_Helper::update('posts', [
				'post_status' => 'draft'
			], [
				'ID' => $post->ID
			], ['%s']);
		}

		return $posts;
	}

	public function unpublish_courses(): ?array
	{
		$posts = Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			INNER JOIN {$this->postmeta} AS pm
			ON pm.post_id = p.ID
			AND pm.meta_key = 'course_endDate'
			AND pm.meta_value < %s
			WHERE p.post_status = 'publish'
		", [Date_Helper::gmt_now(2)]);

		foreach ($posts as $post) {
			Db_Helper::update('posts', [
				'post_status' => 'draft'
			], [
				'ID' => $post->ID
			], ['%s']);
		}

		return $posts;
	}

	public function get_node_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_node_table WHERE post_id = %d", [$post_id]);

		return array_map(fn ($row): int => $row->node_id, $rows);
	}

	public function get_category_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_category_table WHERE post_id = %d", [$post_id]);

		return array_map(fn ($row): int => $row->category_id, $rows);
	}

	public function get_user_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_user_table WHERE post_id = %d", [$post_id]);

		return array_map(fn ($row): int => $row->user_id, $rows);
	}

	public function get_editor_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_editor_table WHERE post_id = %d AND type = 'editor'", [$post_id]);

		return array_map(fn ($row): int => $row->user_id, $rows);
	}
}
