<?php

namespace Rubicore\Core;

use \WP_REST_Request, \WP_User;

final class Comment_Controller
{

	function __construct()
	{
		$this->routes();
	}

	private function routes(): void
	{
		$api = new Api_Helper('rubicore/core/v1');

		$api->add_route('comment', 'GET', function (WP_REST_Request $req, WP_User $user): array {
			User_Helper::allowed_to('comment_list', $user);
			$comment_repo = new Comment_Repository();
			$user_repo = new User_Repository();
			$params = $req->get_params();
			$comments = $comment_repo->get($params);

			return array_map(function ($comment) use($user_repo) : Comment_Model {
				$comment->user = $user_repo->get_by_id($comment->user_id);
				return Comment_Transformer::transform_out($comment);
			}, $comments);
		}, array(
			'args' => array_merge(
				Comment_Query::get_args(),
				Query_Helper::get_pagination_args()
			)
		));

		$api->add_route('comment/(?P<comment_id>\d+)', 'GET', function (WP_REST_Request $req, WP_User $user): Comment_Model {
			User_Helper::allowed_to('comment_get', $user);
			$comment_repo = new Comment_Repository();
			$user_repo = new User_Repository();
			$comment = $comment_repo->get_by_id($req->get_param('comment_id'));

			$comment->user = $user_repo->get_by_id($comment->user_id);

			return Comment_Transformer::transform_out($comment);
		}, array(
			'args' => array(
				'comment_id' => array('type' => Api_Helper::TYPE_INT, 'required' => true, 'minimum' => 1)
			)
		));

		$api->add_route('comment', 'POST', function (WP_REST_Request $req, WP_User $user): int {
			User_Helper::allowed_to('comment_create', $user);

			$body = Request_Helper::get_body($req);
			$post_id = $body->postId;
			$content = $body->content;

			$post = (new Post_Repository())->get_by_slug($post_id, null);
			if (!$post) {
				Response_Helper::not_found("No post found with provided id.");
			}

			if ($post->comment_status !== 'open') {
				Response_Helper::die('Comments disabled for post with provided id.');
			}

			$comment_repo = new Comment_Repository();
			return $comment_repo->create($post_id, $user->ID, $content);
		}, array(
			'args' => array(
				'postId' => array('type' => Api_Helper::TYPE_INT, 'required' => true, 'minimum' => 1),
				'content' => array('type' => Api_Helper::TYPE_STRING, 'required' => true, 'minLength' => 1)
			)
		));

		$api->add_route('comment/(?P<comment_id>\d+)', 'DELETE', function (WP_REST_Request $req, WP_User $user): void {
			$comment_repo = new Comment_Repository();
			$comment = $comment_repo->get_by_id($req->get_param('comment_id'));

			if (!$comment) {
				Response_Helper::not_found("No comment found with provided id.");
			}

			User_Helper::allowed_to('comment_delete', $user, $comment->user_id);

			$comment_repo->delete($req->get_param('comment_id'));
		}, array(
			'args' => array(
				'comment_id' => array('type' => Api_Helper::TYPE_INT, 'required' => true, 'minimum' => 1)
			)
		));

		$api->add_route('comment/(?P<comment_id>\d+)', 'PUT', function (WP_REST_Request $req, WP_User $user): void {
			$comment_repo = new Comment_Repository();
			$comment = $comment_repo->get_by_id($req->get_param('comment_id'));
			$body = Request_Helper::get_body($req);

			if (!$comment) {
				Response_Helper::not_found("No comment found with provided id.");
			}

			User_Helper::allowed_to('comment_edit', $user, $comment->user_id);

			$comment_repo->update($req->get_param('comment_id'), $body->content);
		}, array(
			'args' => array(
				'comment_id' => array('type' => Api_Helper::TYPE_INT, 'required' => true, 'minimum' => 1),
				'content' => array('type' => Api_Helper::TYPE_STRING, 'required' => true, 'minLength' => 1)
			)
		));

	}
}
