<?php

namespace Rubicore\Core;

use \WP_REST_Request, \WP_User;

final class Post_Controller {

	function __construct ()
	{
		$this->routes();
	}

	private function routes() : void
	{
		$api = new Api_Helper('rubicore/core/v1');


		$api->add_route('post-slug-availability/(?P<slug>[^/]+?)', 'GET', function (WP_REST_Request $req, WP_User $user) : array {
			$slug = $req->get_param('slug');

			return ['available' => Post_Service::slug_available($slug)];
		}, [
			'args' => [
				'slug' => ['type' => Api_Helper::TYPE_STRING, 'required' => true]
			]
		]);


		$api->add_route('my-post', 'GET', function (WP_REST_Request $req, WP_User $user): array {
			User_Helper::allowed_to('post_list', $user);

			$post_repo = new Post_Repository();
			$params = $req->get_params();

			$posts = $post_repo->get_by_user_id($params, $user->ID);
			$media_repo = new Media_Repository();
			$user_repo = new User_Repository();

			return array_map(function ($post) use ($req, $media_repo, $user_repo): Post_Model {
				$post->user = $user_repo->get_by_id($post->post_author);
				$post->media = $post->media_id ? $media_repo->get_by_id($post->media_id) : null;
				$post->post_content = $req->get_param('includeContent') ? $post->post_content : null;

				return Post_Transformer::transform_out($post);
			}, $posts);
		}, array(
			'args' => array_merge(
				Post_Query::get_args(),
				Query_Helper::get_pagination_args(),
				array(
					'includeContent' => array(
						'type' => Api_Helper::TYPE_BOOL,
						'required' => false,
						'default' => false
					)
				)
			)
		));

		$api->add_route('post-publish', 'POST', function (WP_REST_Request $req, WP_User $user) : void {
			$post_repo = new Post_Repository();

			$post_repo->publish_posts();
		});


		$api->add_route('post-unpublish', 'POST', function (WP_REST_Request $req, WP_User $user) : void {
			$post_repo = new Post_Repository();

			$post_repo->unpublish_posts();
		});


		$api->add_route('post/(?P<slug>[^/]+?)', 'GET', function (WP_REST_Request $req, WP_User $user) : Post_Model {
			$post = Post_Service::get_by_slug($req->get_param('slug'), $req->get_param('type'));

			if (!$post) {
				Response_Helper::not_found('No post found with provided id.');
			}

			User_Helper::allowed_to('post_get', $user, [$post->author->id, ...$post->editorIds]);

			return $post;
		}, array(
			'args' => array(
				'type' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false, 'default' => null)
			)
		));


		$api->add_route('post', 'GET', function (WP_REST_Request $req, WP_User $user): array {
			User_Helper::allowed_to('post_list', $user);

			$params = $req->get_params();
			$post_service = new Post_Service();

			return $post_service->get($params, $user, [
				'page', 'news', 'pinned-news',
				'page-related', 'info', 'info-important',
				'alert', 'advertisment', 'givetake'
			]);
		},
			array(
				'args' => Post_Service::get_args()
			)
		);


		$api->add_route('post', 'POST', function (WP_REST_Request $req, WP_User $user) : int
		{
			$body = Request_Helper::get_body($req);
			$type_cap = Capability_Helper::get_post_capability($body->type);

			User_Helper::allowed_to($type_cap, $user);

			return Post_Service::post($req, $user);
		}, array(
			'args' => Post_Service::post_args()
		));


		$api->add_route('post/(?P<post_id>[^/]+?)', 'PUT', function (WP_REST_Request $req, WP_User $user): void {
			$body = Request_Helper::get_body($req);
			$body->authorId = !is_null($body->authorId) ? $body->authorId : $user->ID;

			$post_repo = new Post_Repository();
			$post_id = $req->get_param('post_id');
			$post = $post_repo->get_by_slug($post_id, null);

			if (!$post) {
				Response_Helper::not_found('No post found with provided id.');
			}

			User_Helper::allowed_to('post_edit', $user, [$post->post_author, ...$post->editor_ids]);

			Post_Service::update($post_id, $req, $user);
		}, [
			'args' => Post_Service::post_args()
		]);


		$api->add_route('post/(?P<post_id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user): void {
			$post_repo = new Post_Repository();
			$post_id = $req->get_param('post_id');

			$post = $post_repo->get_by_slug($post_id, null);

			if (!$post) {
				Response_Helper::not_found('No post found with provided id.');
			}

			User_Helper::allowed_to('post_delete', $user, [$post->post_author, ...$post->editor_ids]);

			$post_children = $post_repo->get_children($post_id);

			if (count($post_children) != 0) {
				Response_Helper::bad('Unable to delete post with children.');
			}

			$post_repo->delete($post_id);
		},
			array(
				'args' => array(
					'post_id' => array('type' => Api_Helper::TYPE_INT, 'required' => true, 'minimum' => 1)
				)
			)
		);

	}
}
