<?php

namespace Rubicore\Core;
use \WP_REST_Request, \WP_User;

final class Token_Controller {

	function __construct () {
		$this->routes();
	}

	private function routes() : void {
		$api = new Api_Helper('rubicore/core/v1');

		$api->add_route('token', 'GET', function (WP_REST_Request $req, WP_User $user): array {
			User_Helper::allowed_to('token_list', $user);
			$token_repo = new Token_Repository();
			$params = $req->get_params();

			$tokens = $token_repo->get($params, $req->get_param('expired'));

			return array_map(fn($token) : Token_Model => Token_Transformer::transform_out($token), $tokens);
		}, array(
			'args' => array_merge(
				Token_Query::get_args(),
				Query_Helper::get_pagination_args(),
				array(
					'expired' => array(
						'type' => Api_Helper::TYPE_BOOL,
						'required' => false
					)
				)
			)
		));

		$api->add_route('token', 'POST', function (WP_REST_Request $req, WP_User $user): ?Token_Model {
			$type = $req->get_param('type');
			$user_id = $req->get_param('userId');

			if ($type == 'api') {
				User_Helper::allowed_to('token_create_api', $user);
			}

			if ($user_id) {
				User_Helper::allowed_to('token_create', $user);
				$user_repo = new User_Repository();

				if (!$user_repo->get_by_id($user_id)) {
					Response_Helper::not_found('No user found with provided id.');
				}

				$token_repo = new Token_Repository();
				$token = $token_repo->create($user_id, $type);

				return Token_Transformer::transform_out($token);
			}

			if ($req->get_param('username') && $req->get_param('password')) {

				$user = wp_signon(array(
					'user_login' => $req->get_param('username'),
					'user_password' => $req->get_param('password')
				), false);

				if (is_wp_error($user) || $user->ID == 0) {
					Response_Helper::not_authorized('Access denied');
				}

				$token_repo = new Token_Repository();
				$token = $token_repo->create($user->ID, $type);

				return Token_Transformer::transform_out($token);
			} else {
				Response_Helper::bad();
			}

			return null;
		}, array(
			'auth_required' => false,
			'args' => array(
				'username' => array(
					'type' => Api_Helper::TYPE_STRING,
					'minLength' => 1
				),

				'password' => array(
					'type' => Api_Helper::TYPE_STRING,
					'minLength' => 1
				),

				'userId' => array(
					'type' => Api_Helper::TYPE_INT,
					'minimum' => 1
				),

				'type' => array(
					'type' => Api_Helper::TYPE_STRING,
					'required' => false,
					'default' => 'web',
					"enum" => array('web', 'api')
				)
			)
		));


		$api->add_route('token/(?P<id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user): void {
			User_Helper::allowed_to('token_delete', $user);

			$token_repo = new Token_Repository();
			$token = $token_repo->get_by_id($req->get_param('id'));

			if (!$token) {
				Response_Helper::not_found('No token found with provided id.');
			}

			$token_repo->delete($req->get_param('id'));
		});
	}

}
