<?php

namespace Rubicore\Core;

use \WP_REST_Request, \WP_User;

final class Post_Service {

	public static function get(WP_REST_Request $req, WP_User $user, array $post_types)
	{
		$post_repo = new Post_Repository();
		$media_repo = new Media_Repository();
		$user_repo = new User_Repository();

		$filters = array(
			'categoryIds' => $req->get_param('categoryId'),
			'userIds' => $req->get_param('userId'),
			'nodeIds' => $req->get_param('nodeId'),
			'courseAttendeeIds' => $req->get_param('courseAttendeeId'),
			'postTypes' => $post_types
		);


		$posts = $post_repo->get($req, $filters, $user->ID);

		return array_map(function ($post) use ($req, $media_repo, $user_repo): Post_Model {
			$post->user = $user_repo->get_by_id($post->post_author);
			$post->media = $post->media_id ? $media_repo->get_by_id($post->media_id) : null;
			$post->post_content = $req->get_param('includeContent') ? $post->post_content : null;

			$post->course_attendees = $post->post_type == 'course' ? Course_Service::get_attendees_by_post_id($post->ID, null) : null;

			return Post_Transformer::transform_out($post);
		}, $posts);
	}

	public static function get_by_slug(string $slug, ?string $type) : ?Post_Model
	{
		$post_repo = new Post_Repository();

		$post = $post_repo->get_by_slug($slug, $type);

		if (!$post) {
			Response_Helper::not_found('No post found with provided id.');
		}

		$post->user = (new User_Repository())->get_by_id($post->post_author);
		$post->media = $post->media_id ? (new Media_Repository())->get_by_id($post->media_id) : null;
		$post->course_attendees = $post->post_type == 'course' ? Course_Service::get_attendees_by_post_id($post->ID, null) : null;

		return Post_Transformer::transform_out($post);
	}

	public static function post(WP_REST_Request $req, WP_User $user) : int
	{
		$post_repo = new Post_Repository();
		$body = Request_Helper::get_body($req);
		$body->authorId = !is_null($body->authorId) ? $body->authorId : $user->ID;

		if ($req->get_param('mediaId') && !$post_repo->get_by_slug($req->get_param('mediaId'), null)) {
			Response_Helper::not_found('No media found with provided id.');
		}

		if ($req->get_param('parentId') && !$post_repo->get_by_slug($req->get_param('parentId'), null)) {
			Response_Helper::not_found('No parent post found with provided id.');
		}

		$user_repo = new User_Repository();
		if ($req->get_param('authorId') && !$user_repo->get_by_id($req->get_param('authorId'))) {
			Response_Helper::not_found('No author found with provided id.');
		}

		$body->order = self::update_order($body, null);

		$post_id = $post_repo->create($body);

		self::update_related($post_id, $body->userIds, 'user_id', new Post_User_Repository());
		self::update_related($post_id, $body->nodeIds, 'node_id', new Post_Node_Repository());
		self::update_related($post_id, $body->categoryIds, 'category_id', new Post_Category_Repository());
		self::update_related($post_id, $body->editorIds, 'user_id', new Post_Editor_Repository(), 'editor');

		do_action('rubicore_core_post_create', [
			'user' => $user,
			'post_id' => $post_id,
			'request' => $req
		]);

		return $post_id;
	}

	public static function update(int $post_id, WP_REST_Request $req, WP_User $user) : void
	{
		$post_repo = new Post_Repository();

		$body = Request_Helper::get_body($req);
		$body->order = self::update_order($body, $post_id);

		$post_repo->update($post_id, $body);

		self::update_related($post_id, $body->userIds, 'user_id', new Post_User_Repository());
		self::update_related($post_id, $body->nodeIds, 'node_id', new Post_Node_Repository());
		self::update_related($post_id, $body->categoryIds, 'category_id', new Post_Category_Repository());
		self::update_related($post_id, $body->editorIds, 'user_id', new Post_Editor_Repository(), 'editor');
	}

	private static function update_related(?int $post_id, ?array $input_ids, string $id_fields, $repo, ?string $type = null): void
	{
		if (is_null($input_ids) || !$post_id) {
			return;
		}

		$current_ids = array_column($repo->get_by_post_id($post_id, $type), $id_fields);
		$to_delete = array_reduce($current_ids, fn ($acc, $id) => !in_array($id, $input_ids) ? [...$acc, $id] : $acc, []);
		$to_add = array_reduce($input_ids, fn ($acc, $id) => !in_array($id, $current_ids) ? [...$acc, $id] : $acc, []);

		array_walk($to_delete, fn ($id) => $repo->remove($post_id, $id, $type));
		array_walk($to_add, fn ($id) => $repo->add($post_id, $id, $type));
	}

	private static function update_order(object $body, ?int $post_id): int
	{
		$new_order = 0;
		$post_repo = new Post_Repository();

		if ($body->orderPostBeforeId) {
			$before_item = $post_repo->get_by_slug($body->orderPostBeforeId, null);
		} else {
			$before_item = new \stdClass();
			$before_item->menu_order = -2;
		}

		$sort_posts = $post_repo->get_sort_list($body->parentId ?? 0, $before_item->menu_order, $body->type, $post_id);
		$new_order = $before_item->menu_order + 1;

		$posts_to_update = Post_Helper::sort($new_order, $sort_posts);
		$post_repo->update_sort($posts_to_update);

		return $new_order;
	}


	public static function get_args()
	{
		return array_merge(
			Post_Query::get_args(),
			Query_Helper::get_pagination_args(),
			[
				'categoryId' => [
					'in' => Api_Helper::IN_QUERY,
					'type' => Api_Helper::TYPE_ARRAY,
					'required' => false,
					'uniqueItems' => true,
					'items' => [
						'type' => Api_Helper::TYPE_INT,
						'minimum' => 1
					]
				],
				'userId' => [
					'in' => Api_Helper::IN_QUERY,
					'type' => Api_Helper::TYPE_ARRAY,
					'required' => false,
					'uniqueItems' => true,
					'items' => [
						'type' => Api_Helper::TYPE_INT,
						'minimum' => 1
					]
				],
				'nodeId' => [
					'in' => Api_Helper::IN_QUERY,
					'type' => Api_Helper::TYPE_ARRAY,
					'required' => false,
					'uniqueItems' => true,
					'items' => [
						'type' => Api_Helper::TYPE_INT,
						'minimum' => 1
					]
				],
				'courseAttendeeId' => [
					'in' => Api_Helper::IN_QUERY,
					'type' => Api_Helper::TYPE_ARRAY,
					'required' => false,
					'uniqueItems' => true,
					'items' => [
						'type' => Api_Helper::TYPE_INT,
						'minimum' => 1
					]
				],
				'includeContent' => [
					'in' => Api_Helper::IN_QUERY,
					'type' => Api_Helper::TYPE_BOOL,
					'required' => false,
					'default' => false
				]
			]
		);
	}


	public static function post_args() : array
	{
		return array(
			'type' => array('type' => Api_Helper::TYPE_STRING, 'required' => true, 'minLength' => 1, 'maxLength' => 20),
			'title' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
			'excerpt' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
			'content' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
			'mediaId' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_INT], 'required' => false, 'minimum' => 1),
			'authorId' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_INT], 'required' => false, 'minimum' => 1),
			'parentId' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_INT], 'required' => false, 'minimum' => 1),
			'orderPostBeforeId' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_INT], 'required' => false, 'minimum' => 1),
			'publishedDate' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_STRING], 'required' => false, 'format' => 'date-time'),
			'userIds' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_ARRAY], 'required' => false, 'items' => array(
				'type' => Api_Helper::TYPE_INT,
				'minimum' => 1
			)),
			'nodeIds' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_ARRAY], 'required' => false, 'items' => array(
				'type' => Api_Helper::TYPE_INT,
				'minimum' => 1
			)),
			'categoryIds' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_ARRAY], 'required' => false, 'items' => array(
				'type' => Api_Helper::TYPE_INT,
				'minimum' => 1
			)),
			'editorIds' => array('type' => [Api_Helper::TYPE_NULL, Api_Helper::TYPE_ARRAY], 'required' => false, 'items' => array(
				'type' => Api_Helper::TYPE_INT,
				'minimum' => 1
			)),
			'meta' => array('type' => Api_Helper::TYPE_OBJECT, 'required' => false, 'properties' => array(
				'blurbs' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false),
				'info' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false),
				'contact' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false),
				'unpublishedDate' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false, 'format' => 'date-time'),
				'authoredDate' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false, 'format' => 'date-time'),
				'pseudoAuthor' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false),
				'courseStartDate' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false),
				'courseEndDate' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false, 'format' => 'date-time'),
				'courseCancellationDate' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false, 'format' => 'date-time'),
				'coursePlace' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false),
				'courseNumberOfAttendees' => array('type' => [Api_Helper::TYPE_INT, Api_Helper::TYPE_NULL], 'required' => false),
				'courseBookable' => array('type' => [Api_Helper::TYPE_BOOL, Api_Helper::TYPE_NULL], 'required' => false),
				'courseAttendeeInformation' => array('type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL], 'required' => false)
			))
		);
	}
}

