<?php

namespace Rubicore\Core;
use \WP_REST_Request;

final class Query_Helper {

	public static function get_sort_query(?array $sort = [], array $fields) : string {
		if (!$sort) {
			return "";
		}

		$arr = array();

		foreach ($sort as $value) {
			$type = str_contains($value, 'ASC') ? 'ASC' : 'DESC';
			$trimmed = str_replace(' ASC', '', $value);
			$trimmed = str_replace(' DESC', '', $trimmed);

			$field_key = array_search($trimmed, array_column($fields, 'key'));

			if ($field_key === false) {
				continue;
			}

			$trimmed = $fields[$field_key]['internal_key'];

			//$str = preg_replace('~[^a-z_-]+~i', '', $trimmed);

			$arr[] = "$trimmed $type";
		}

		if (count($arr) == 0) {
			return "";
		}

		return Db_Helper::prepare("ORDER BY " . implode(',', $arr));
	}

	public static function get_count(string $query) : string {
		return "SELECT COUNT(*) AS num_rows $query";
	}

	public static function get_limit_query(WP_REST_Request $request) : string {
		$query_params = $request->get_query_params();

		$page = isset($query_params['page']) ?
			intval($query_params['page']) :
			null;

		$per_page = isset($query_params['perPage']) ?
			intval($query_params['perPage']) :
			null;

		if (is_null($page) || is_null($per_page)) {
			return "";
		}

		if ($page < 1) {
			$page = 1;
		}

		if ($per_page < 1) {
			$per_page = 10;
		}

		$offset = $page == 1 ? 0 : ($page - 1) * $per_page;

		return Db_Helper::prepare("LIMIT %d,%d", [$offset, $per_page]);
	}

	public static function get_search_query(?string $search, array $fields) : string {
		if (!$search || count($fields) == 0) {
			return "";
		}

		$query = "";

		foreach ($fields as $key => $value) {
			$query .= $key == '0' ?
				Db_Helper::prepare("{$value['internal_key']} LIKE '%%%s%%'", [$search]) :
				Db_Helper::prepare(" OR {$value['internal_key']} LIKE '%%%s%%'", [$search]);
		}

		return "($query)";
	}

	public static function get_filter_query($params, array $fields) : string {
		$query = "";
		$is_first = true;

		foreach ($fields as $value) {
			$input_value = isset($params[$value['key']]) ? $params[$value['key']] : null;

			if (!$input_value) {
				continue;
			}

			if (is_array($input_value)) {
				$sub_query = "";

				foreach ($input_value as $val) {
					$sub_query .= Db_Helper::prepare("{$value['internal_key']} = %s OR ", [$val]);
				}

				$sub_query = rtrim($sub_query, ' OR ');

				if ($is_first) {
					$query .= "({$sub_query})";
					$is_first = false;
				} else {
					$query .= "AND ({$sub_query})";
				}

				continue;
			}


			if ($is_first) {
				$query .= Db_Helper::prepare("{$value['internal_key']} = %s", [$params[$value['key']]]);
				$is_first = false;
			} else {
				$query .= Db_Helper::prepare(" AND {$value['internal_key']} = %s", [$params[$value['key']]]);
			}

		}

		return $query;
	}


	public static function add_where($must_be_true, string $current_where, string $type, string $where) : string {
		if (boolval($must_be_true) === false && $must_be_true !== 0) {
			return $current_where;
		}

		$operator = "WHERE";

		if (str_starts_with($current_where, "WHERE")) {
			$operator = $type;
		}

		return $current_where === "" ? "$operator $where" : "$current_where $operator $where";
	}

	public static function get_sortables(array $fields) : array {
		$ret = [];
		foreach ($fields as $field) {
			if ($field['sortable']) {
				$ret[] = $field;
			}
		}

		return $ret;
	}

	public static function get_filterables(array $fields): array {
		$ret = [];
		foreach ($fields as $field) {
			if ($field['filterable']) {
				$ret[] = $field;
			}
		}

		return $ret;
	}

	public static function get_searchables(array $fields): array {
		$ret = [];
		foreach ($fields as $field) {
			if ($field['searchable']) {
				$ret[] = $field;
			}
		}

		return $ret;
	}

	public static function generate_args(array $fields) {
		$args = array();
		$sort = self::get_sort_args($fields);
		$search = self::get_search_args($fields);

		if ($sort) {
			$args = array_merge($args, $sort);
		}

		if ($search) {
			$args = array_merge($args, $search);
		}

		return array_merge($args, self::get_filter_args($fields));
	}

	public static function get_search_args(array $fields) : ?array {
		$ret = null;

		$rows = array_filter($fields, fn($val) => $val['searchable'] == true);

		if (count($rows) !== 0) {
			$ret = array('search' => array(
				'type' => Api_Helper::TYPE_STRING,
				'in' => Api_Helper::IN_QUERY,
				'required' => false
			));
		}

		return $ret;
	}

	public static function get_sort_args(array $fields) : ?array {
		$ret = null;
		$sort_keys = [];

		foreach($fields as $field) {
			if ($field['sortable']) {
				$sort_keys[] = "{$field['key']} DESC";
				$sort_keys[] = "{$field['key']} ASC";
			}
		}

		if (count($sort_keys) !== 0) {
			$ret = array('sort' => array(
				'type' => Api_Helper::TYPE_ARRAY,
				'in' => Api_Helper::IN_QUERY,
				'required' => false,
				'uniqueItems' => true,
				'items' => array(
					'type' => Api_Helper::TYPE_STRING,
					'enum' => $sort_keys
				)
			));
		}

		return $ret;
	}

	public static function get_filter_args(array $fields): array {
		$ret = [];

		foreach ($fields as $field) {
			if ($field['filterable']) {
				$ret[$field['key']] = array(
					'type' => Api_Helper::TYPE_ARRAY,
					'in' => Api_Helper::IN_QUERY,
					'items' => array(
						'type' => $field['type']
					)
				);
			}
		}

		return $ret;
	}

	public static function get_pagination_args() : array {
		return array(
			'page' => array(
				'type' => Api_Helper::TYPE_INT,
				'in' => Api_Helper::IN_QUERY,
				'required' => false,
				'default' => 1,
				'minimum' => 1
			),
			'perPage' => array(
				'type' => Api_Helper::TYPE_INT,
				'in' => Api_Helper::IN_QUERY,
				'required' => false,
				'default' => 10,
				'minimum' => 1
			)
		);
	}
}
