<?php

namespace Rubicore\Core;
use \WP_REST_Request, \WP_User;

final class Link_Controller {

	private Link_Service $_linkService;

	function __construct () {
		$this->_linkService = new Link_Service();
		$this->routes();
	}


	private function routes() : void {
		$api = new Api_Helper('rubicore/core/v1');

		$api->add_route('link', 'GET', function (WP_REST_Request $req, WP_User $user) : array {
			User_Helper::allowed_to('get_links', $user);

			$link_repo = new Link_Repository();
			$user_repo = new User_Repository();
			$params = $req->get_params();

			$links = $link_repo->get($params, $req->get_param('userId'));

			return array_map(function($link) use($user_repo, $link_repo) : Link_Model {
				$link->user = $user_repo->get_by_id($link->author);
				$link->nodes = $link_repo->get_nodes($link->id);
				return Link_Transformer::transform_out($link);
			}, $links);
		}, array(
			'args' => array_merge(
				array(
					'userId' => array('type' => Api_Helper::TYPE_INT, 'required' => false, 'minimum' => 1)
				),
				Link_Query::get_args(),
				Query_Helper::get_pagination_args()
			)
		));



		$api->add_route('link/(?P<link_id>[^/]+?)', 'GET', function (WP_REST_Request $req, WP_User $user) : Link_Model {
			$link_repo = new Link_Repository();
			$link_id = $req->get_param('link_id');
			$link = $link_repo->get_by_id($link_id);

			if (!$link) {
				Response_Helper::not_found('No link found with provided id.');
			}

			User_Helper::allowed_to('get_links', $user, intval($link->author));

			$link->nodes = $link_repo->get_nodes($link_id);
			$user_repo = new User_Repository();
			$link->user = $user_repo->get_by_id($link->author);

			return Link_Transformer::transform_out($link);
		}, array(
			'args' => array(
				'link_id' => array('type' => Api_Helper::TYPE_INT, 'required' => false, 'minimum' => 1)
			)
		));



		$api->add_route('link/(?P<link_id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user) : void {
			$link_repo = new Link_Repository();
			$link_id = $req->get_param('link_id');
			$link = $link_repo->get_by_id($link_id);

			if (!$link) {
				Response_Helper::not_found('No link found with provided id.');
			}

			User_Helper::allowed_to('delete_links', $user, intval($link->author));

			$link_repo->delete_link($link_id);
		}, array(
			'args' => array(
				'link_id' => array('type' => Api_Helper::TYPE_INT, 'required' => false, 'minimum' => 1)
			)
		));



		$api->add_route('link/(?P<link_id>[^/]+?)', 'PUT', function (WP_REST_Request $req, WP_User $user) : void {
			$link_id = $req->get_param('link_id');
			$body = Request_Helper::get_body($req);

			$this->_linkService->Update($link_id, $body, $user);
		}, array(
			'args' => array(
				'link_id' => array('type' => Api_Helper::TYPE_INT, 'required' => false, 'minimum' => 1),
				'name' => array('type' => Api_Helper::TYPE_STRING, 'minLength' => 2, 'required' => true),
				'type' => array('type' => Api_Helper::TYPE_STRING, 'minLength' => 3, 'required' => true),
				'url' => array('type' => Api_Helper::TYPE_STRING, 'minLength' => 10, 'required' => true),
				'nodeIds' => array('type' => Api_Helper::TYPE_ARRAY, 'required' => false, 'items' => array(
					'type' => Api_Helper::TYPE_INT, 'required' => true, 'minimum' => 1
				)),
				'shortDescription' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'longDescription' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'icon' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'author' => array('type' => [Api_Helper::TYPE_INT, Api_Helper::TYPE_NULL], 'required' => false, 'minimum' => 1),
				'public' => array('type' => Api_Helper::TYPE_BOOL, 'required' => false, 'default' => true),
				'sort' => array('type' => Api_Helper::TYPE_INT, 'required' => false)
			)
		));



		$api->add_route('link', 'POST', function (WP_REST_Request $req, WP_User $user) : int {
			User_Helper::allowed_to('post_links', $user);

			$body = Request_Helper::get_body($req);

			return $this->_linkService->Create($body, $user->ID);
		}, array(
			'args' => array(
				'name' => array('type' => Api_Helper::TYPE_STRING, 'minLength' => 2, 'required' => true),
				'type' => array('type' => Api_Helper::TYPE_STRING, 'minLength' => 3, 'required' => true),
				'url' => array('type' => Api_Helper::TYPE_STRING, 'minLength' => 10, 'required' => true),
				'nodeIds' => array('type' => Api_Helper::TYPE_ARRAY, 'required' => false, 'items' => array(
					'type' => Api_Helper::TYPE_INT, 'required' => true, 'minimum' => 1
				)),
				'shortDescription' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'longDescription' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'icon' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'author' => array('type' => [Api_Helper::TYPE_INT, Api_Helper::TYPE_NULL], 'required' => false, 'minimum' => 1),
				'public' => array('type' => Api_Helper::TYPE_BOOL, 'required' => false, 'default' => true),
				'sort' => array('type' => Api_Helper::TYPE_INT, 'required' => false)
			)
		));

	}
}
