<?php

namespace Rubicore\Core;

use \WP_REST_Request, \WP_User;

final class Category_Controller {

	function __construct () {
		$this->routes();
	}

	private function routes() : void {
		$api = new Api_Helper('rubicore/core/v1');

		$api->add_route('category', 'GET', function (WP_REST_Request $req, WP_User $user) : array {
			$category_repo = new Category_Repository();
			$params = $req->get_params();
			$filter = array(
				'userId' => $req->get_param('userId')
			);

			$categories = $category_repo->get($params, $filter);

			return array_map(fn($category) : Category_Model => Category_Transformer::transform_out($category), $categories);
		}, array(
			'args' => array_merge(
				array(
					'userId' => array('type' => Api_Helper::TYPE_INT, 'required' => false, 'minimum' => 1)
				),
				Category_Query::get_args(),
				Query_Helper::get_pagination_args()
			)
		));



		$api->add_route('category/(?P<id>[^/]+?)', 'GET', function (WP_REST_Request $req, WP_User $user) : Category_Model {
			$repo = new Category_Repository();
			$category = $repo->get_by_id($req->get_param('id'));

			if (!$category) {
				Response_Helper::not_found('No category found with provided id.');
			}

			return Category_Transformer::transform_out($category);
		}, [
			'args' => [
				'id' => ['type' => [Api_Helper::TYPE_INT, Api_Helper::TYPE_STRING], 'required' => true]
			]
		]);



		$api->add_route('category', 'POST', function (WP_REST_Request $req, WP_User $user) : int {
			$entity = Category_Transformer::transform_in(Request_Helper::get_body($req));
			$repo = new Category_Repository();

			return $repo->create($entity);
		}, array(
			'args' => array(
				'type' => array('type' => Api_Helper::TYPE_STRING, 'required' => true, 'minLength' => 1),
				'name' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'key' => array('type' => Api_Helper::TYPE_STRING, 'required' => true, 'minLength' => 1)
			)
		));



		$api->add_route('category/(?P<id>[^/]+?)', 'PUT', function (WP_REST_Request $req, WP_User $user) : void {
			$body = Request_Helper::get_body($req);
			$body->id = $req->get_param('id');

			$entity = Category_Transformer::transform_in($body);
			$repo = new Category_Repository();
			$category = $repo->get_by_id($entity->id);

			if (!$category) {
				Response_Helper::not_found('No category found with provided id.');
			}

			$repo->update($req->get_param('id'), $entity);
		}, array(
			'args' => array(
				'type' => array('type' => Api_Helper::TYPE_STRING, 'required' => true, 'minLength' => 1),
				'name' => array('type' => Api_Helper::TYPE_STRING, 'required' => false),
				'key' => array('type' => Api_Helper::TYPE_STRING, 'required' => true, 'minLength' => 1)
			)
		));



		$api->add_route('category/(?P<id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user) : void {
			$repo = new Category_Repository();
			$category = $repo->get_by_id($req->get_param('id'));

			if (!$category) {
				Response_Helper::not_found('No category found with provided id.');
			}

			$repo->delete($req->get_param('id'));
		});



		$api->add_route('category/(?P<id>[^/]+?)/node/(?P<item_id>[^/]+?)', 'POST', function (WP_REST_Request $req, WP_User $user) : void {
			$category_repo = new Category_Repository();
			$category = $category_repo->get_by_id($req->get_param('id'));


			if (!$category) {
				Response_Helper::not_found('No category found with provided id.');
			}

			$node_repo = new Node_Repository();
			$node = $node_repo->get_by_id($req->get_param('item_id'));

			if (!$node) {
				Response_Helper::not_found('No node found with provided id.');
			}

			$category_node_repo = new Category_Node_Repository();
			$category_node_repo->add($req->get_param('id'), $req->get_param('item_id'));
		});



		$api->add_route('category/(?P<id>[^/]+?)/node/(?P<item_id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user) : void {
			$category_repo = new Category_Repository();
			$category = $category_repo->get_by_id($req->get_param('id'));

			if (!$category) {
				Response_Helper::not_found('No category found with provided id.');
			}

			$node_repo = new Node_Repository();
			$node = $node_repo->get_by_id($req->get_param('item_id'));

			if (!$node) {
				Response_Helper::not_found('No node found with provided id.');
			}

			$category_node_repo = new Category_Node_Repository();
			$category_node_repo->remove($req->get_param('id'), $req->get_param('item_id'));
		});

	}

}
