<?php

namespace Rubicore\Core;

use \WP_REST_Request, \WP_User;

final class Course_Service
{


	public static function get_by_post(object $post) : ?Course_Model
	{
		if ($post->post_type != 'course') {
			return null;
		}

		return Course_Transformer::transform_out($post, self::get_attendees_by_post_id($post->ID));
	}

	public static function get_my(int $user_id, array $params) : array {
		$user_repo = new User_Repository();
		$media_repo = new Media_Repository();
		$course_repo = new Course_Repository();

		$posts = $course_repo->get_by_attendee_id($params, $user_id);

		return array_map(function($post) use($user_repo, $media_repo) : Post_Model {
			$post->user = $user_repo->get_by_id($post->post_author);
			$post->media = $post->media_id ? $media_repo->get_by_id($post->media_id) : null;
			$post->course_attendees = Course_Service::get_attendees_by_post_id($post->ID, null);

			return Post_Transformer::transform_out($post);
		} , $posts);
	}

	public static function get_attendees_by_post_id(int $post_id) : array
	{
		$course_repo = new Course_Repository();
		$user_repo = new User_Repository();

		$attendees = $course_repo->get_by_post_id($post_id, null);

		return array_map(function ($attendee) use ($user_repo) {
			$attendee->user = $user_repo->get_by_id($attendee->user_id);

			return Course_Transformer::transform_attendee_out($attendee);
		}, $attendees);
	}



	public static function update_attendee(int $post_id, int $user_id, object $body, WP_User $user) : void
	{
		$course_repo = new Course_Repository();

		$course_attendees = $course_repo->get_by_post_id($post_id, $user_id);

		if (!$course_attendees) {
			Response_Helper::not_found("No user or course found with provided ids.");
		}

		$post = Post_Service::get_by_slug($post_id, "course");

		User_Helper::allowed_to('course_attendee_update', $user, [$post->author->id, ...$post->editorIds]);

		if (!self::date_has_passed($post->course->startDate)) {
			Response_Helper::die("Unable to update attendee information for a started course.");
		}

		$course_repo->update_attendee($post_id, $user_id, $body);
	}

	public static function add_attendee_external(int $post_id, string $name, string $email, object $body): void
	{
		$existing_user = get_user_by('email', $email);

		if (!$existing_user) {
			$uid = Format_Helper::create_uuid();

			$userdata = array(
				'user_login' => $uid,
				'user_nicename' => $uid,
				'user_email' => $email,
				'display_name' => $name,
				'nickname' => $uid,
				'first_name' => $name,
				'last_name' => $name,
				'user_pass' => wp_hash_password(Format_Helper::create_uuid()),
				'meta_input' => [
					'disable_login' => 1,
					'external' => 1
				]
			);
		}

		$user_id = !$existing_user ? wp_insert_user($userdata) : $existing_user->ID;

		$course_repo = new Course_Repository();

		$course_attendees = $course_repo->get_by_post_id($post_id, $user_id);

		if ($course_attendees) {
			Response_Helper::not_found("User is already added to the course.");
		}

		$post = Post_Service::get_by_slug($post_id, "course");

		if (!$post->course->bookable) {
			Response_Helper::die("Unable to add attendee to an unbookable course.");
		}

		if (!self::date_has_passed($post->course->startDate)) {
			Response_Helper::die("Unable to update attendee information for a started course.");
		}

		$course_repo->add_attendee($post_id, $user_id, $body);

		self::update_reserves($post_id);
	}


	public static function add_attendee(int $post_id, int $user_id, object $body) : void
	{
		$course_repo = new Course_Repository();

		$course_attendees = $course_repo->get_by_post_id($post_id, $user_id);

		if ($course_attendees) {
			Response_Helper::not_found("User is already added to the course.");
		}

		$post = Post_Service::get_by_slug($post_id, "course");

		if (!$post->course->bookable) {
			Response_Helper::die("Unable to add attendee to an unbookable course.");
		}

		if (!self::date_has_passed($post->course->startDate)) {
			Response_Helper::die("Unable to update attendee information for a started course.");
		}

		$course_repo->add_attendee($post_id, $user_id, $body);

		self::update_reserves($post_id);
	}



	public static function remove_attendee(int $post_id, int $user_id, WP_User $user) : void
	{
		$course_repo = new Course_Repository();

		$course_attendees = $course_repo->get_by_post_id($post_id, $user_id);

		if (!$course_attendees) {
			Response_Helper::not_found("No user or course found with provided ids.");
		}

		$post = Post_Service::get_by_slug($post_id, "course");

		User_Helper::allowed_to('course_attendee_remove', $user, [$user_id, $post->author->id, ...$post->editorIds]);

		if (!self::date_has_passed($post->course->startDate)) {
			Response_Helper::die("Unable to remove attendee from a started course.");
		}

		$course_repo->remove_attendee($post_id, $user_id);

		self::update_reserves($post_id);
	}



	public static function update_reserves(int $post_id) : void
	{
		$course_repo = new Course_Repository();
		$post = Post_Service::get_by_slug($post_id, "course");
		$free_slots = $post->course->numberOfAttendees - count(array_filter($post->course->attendees, fn($val) => $val->status == 'approved'));

		foreach ($post->course->attendees as $attendee) {
			if ($attendee->status == 'reserve' && $free_slots > 0) {
				$body = new \stdClass();
				$body->information = $attendee->information;
				$body->status = 'approved';

				$course_repo->update_attendee($post_id, $attendee->user->id, $body);

				do_action('rubicore_core_course_approved', [
					'user_id' => $attendee->user->id,
					'post_id' => $post->id
				]);

				$free_slots = $free_slots - 1;
			}
		}
	}



	private static function date_has_passed(string $date) : bool
	{
		return $date > Date_Helper::gmt_now(1);
	}



	public static function attendee_post_args() : array
	{
		return [
			'post_id' => [
				'in' => Api_Helper::IN_PATH,
				'type' => Api_Helper::TYPE_INT,
				'required' => true,
				'minimum' => 1
			],

			'user_id' => [
				'in' => Api_Helper::IN_PATH,
				'type' => Api_Helper::TYPE_INT,
				'required' => true,
				'minimum' => 1
			],

			'information' => [
				'in' => Api_Helper::IN_BODY,
				'type' => Api_Helper::TYPE_STRING,
				'required' => false
			]
		];
	}



	public static function attendee_put_args(): array
	{
		return [
			'post_id' => [
				'in' => Api_Helper::IN_PATH,
				'type' => Api_Helper::TYPE_INT,
				'required' => true,
				'minimum' => 1
			],

			'user_id' => [
				'in' => Api_Helper::IN_PATH,
				'type' => Api_Helper::TYPE_INT,
				'required' => true,
				'minimum' => 1
			],

			'information' => [
				'in' => Api_Helper::IN_BODY,
				'type' => Api_Helper::TYPE_STRING,
				'required' => false
			]
		];
	}



	public static function post_args(): array
	{
		return [
			'course' => [
				'type' => Api_Helper::TYPE_OBJECT,
				'required' => true,
				'properties' => [
					'startDate' => [
						'type' => Api_Helper::TYPE_STRING,
						'required' => true,
						'format' => 'date-time'
					],
					'endDate' => [
						'type' => Api_Helper::TYPE_STRING,
						'required' => true,
						'format' => 'date-time'
					],
					'cancellationDate' => [
						'type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL],
						'required' => false,
						'format' => 'date-time'
					],
					'place' => [
						'type' => Api_Helper::TYPE_STRING,
						'required' => true
					],
					'numberOfAttendees' => [
						'type' => [Api_Helper::TYPE_INT, Api_Helper::TYPE_NULL],
						'required' => false
					],
					'bookable' => [
						'type' => [Api_Helper::TYPE_BOOL, Api_Helper::TYPE_NULL],
						'required' => false
					],
					'attendeeInformation' => [
						'type' => [Api_Helper::TYPE_STRING, Api_Helper::TYPE_NULL],
						'required' => false
					]
				]
			]
		];
	}

}
