<?php

namespace Rubicore\Core;

final class Media_Repository {
	private string $media_dir = 'wp-content/uploads/';
	private string $post_table;
	private string $post_node_table;

	function __construct() {
		$this->post_table = Db_Helper::table('posts');
		$this->post_node_table = Db_Helper::table('post_node');
	}

	public function get_by_id(int $id) : ?object {
		$user_repo = new User_Repository();
		$row = Db_Helper::get_row("SELECT * FROM $this->post_table AS l WHERE l.id = %d AND l.post_type = 'media'", [$id]);

		if ($row) {
			$row->user = $user_repo->get_by_id($row->post_author);
		}

		return $row;
	}

	public function get(array $params, array $filter) : array {
		$q = new Query_Builder_Helper($params, Media_Query::$fields);
		$user_repo = new User_Repository();
		$where = "WHERE p.post_type = 'media'";
		$where = Query_Helper::add_where($q->filter_query, $where, "AND", $q->filter_query);
		$where = Query_Helper::add_where($q->search_query, $where, "AND", $q->search_query);


		if ($filter['nodeIds']) {
			$node_ids = implode(',', $filter['nodeIds']);

			$where = Query_Helper::add_where(true, $where, "AND", "
				EXISTS (
					SELECT * FROM $this->post_node_table AS pci
					WHERE pci.post_id = p.ID
					AND pci.node_id IN ($node_ids)
				)
			");
		}

		$count_query = Query_Helper::get_count("FROM $this->post_table AS p $where");
		$total_count = Db_Helper::get_row($count_query)->num_rows;
		Response_Helper::set_pagination_total_header($total_count);

		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_table AS p $where $q->order_by_query $q->limit_query");

		foreach ($rows as $row) {
			$row->user = $user_repo->get_by_id($row->post_author);
		}

		return $rows;
	}

	public function create($file, $content_type, $post_data) : ?int {
		$filetype = $content_type['subtype'] == 'jpeg' ? 'jpg' : $content_type['subtype'];
		$filename = Format_Helper::create_uuid();
		$post_data->filename = "{$filename}.{$filetype}";
		$media_entity = Media_Transformer::transform_in($post_data);

		$full_path = "{$this->media_dir}{$post_data->filename}";
		$full_path_thumb = "{$this->media_dir}{$filename}_thumb.{$filetype}";


		if (!file_put_contents($full_path, $file)) {
			Response_Helper::die('Unable to save file content to disk.', 500);
		}

		$this->create_thumb($full_path, $full_path_thumb, $filetype);

		return wp_insert_post($media_entity, true);
	}

	public function delete(int $id, string $filename) : void {
		$ext = pathinfo($filename, PATHINFO_EXTENSION);
		$filename_thumb = str_replace(".$ext", "_thumb.$ext", $filename);
		$full_path = "{$this->media_dir}{$filename}";
		$full_path_thumb = "{$this->media_dir}{$filename_thumb}";

		wp_delete_post($id, true);


		if (file_exists($full_path)) {
			unlink($full_path);
		}

		if (file_exists($full_path_thumb)) {
			unlink($full_path_thumb);
		}
	}


	private function create_thumb($source, $destImagePath, $filetype, $thumbWidth = 250) {
		if ($filetype === 'jpg') {
			$sourceImage = \imagecreatefromjpeg($source);
		}

		if ($filetype === 'png') {
			$sourceImage = \imagecreatefrompng($source);
		}

		if ($filetype === 'gif') {
			$sourceImage = \imagecreatefromgif($source);
		}

		if ($filetype === 'webp') {
			$sourceImage = \imagecreatefromwebp($source);
		}

		$orgWidth = \imagesx($sourceImage);
		$orgHeight = \imagesy($sourceImage);
		$thumbHeight = floor($orgHeight * ($thumbWidth / $orgWidth));
		$destImage = \imagecreatetruecolor($thumbWidth, $thumbHeight);

		\imagecopyresampled($destImage, $sourceImage, 0, 0, 0, 0, $thumbWidth, $thumbHeight, $orgWidth, $orgHeight);

		if ($filetype === 'jpg') {
			\imagejpeg($destImage, $destImagePath);
		}

		if ($filetype === 'png') {
			\imagepng($destImage, $destImagePath);
		}

		if ($filetype === 'gif') {
			\imagegif($destImage, $destImagePath);
		}

		if ($filetype === 'webp') {
			\imagewebp($destImage, $destImagePath);
		}

		\imagedestroy($sourceImage);
		\imagedestroy($destImage);
	}

}
