<?php

namespace Rubicore\Core;

final class Post_Repository {
	private string $post_table;
	private string $post_category_table;
	private string $post_node_table;
	private string $post_user_table;
	private string $post_editor_table;
	private string $course_user_table;
	private string $postmeta;

	function __construct () {
		$this->post_table = Db_Helper::table('posts');
		$this->post_category_table = Db_Helper::table('post_category');
		$this->post_node_table = Db_Helper::table('post_node');
		$this->post_user_table = Db_Helper::table('post_user');
		$this->post_editor_table = Db_Helper::table('post_editor');
		$this->postmeta = Db_Helper::table('postmeta');
		$this->course_user_table = Db_Helper::table('course_user');
	}

	public function get_by_slug($input, ?string $type) : ?object {
		$query = is_numeric($input) ?
			Db_Helper::prepare("SELECT * FROM $this->post_table AS l WHERE l.id = %d", [$input]) :
			Db_Helper::prepare("SELECT * FROM $this->post_table AS l WHERE l.post_name = %s", [$input]);

		$post = $type ?
			Db_Helper::get_row("{$query} AND l.post_type = %s", [$type]) :
			Db_Helper::get_row($query);

		if (!$post) {
			return null;
		}

		$meta = get_post_meta($post->ID);
		$post->meta = [];

		foreach ($meta as $key => $value) {
			$post->meta[$key] = $value[0];
		}

		$post->media_id = isset($post->meta['main_media_id']) ? intval($post->meta['main_media_id']) : null;
		$post->node_ids = $this->get_node_ids($post->ID);
		$post->category_ids = $this->get_category_ids($post->ID);
		$post->user_ids = $this->get_user_ids($post->ID);
		$post->editor_ids = $this->get_editor_ids($post->ID);

		return $post;
		// TODO: Make sure you can't get unpublished posts if you're not the author/editor
	}


	public function get(\WP_REST_Request $req, array $filter): array
	{
		$q = new Query_Builder_Helper($req, Post_Query::$fields);

		$category_ids = implode(',', $filter['categoryIds'] ?: [0]);
		$node_ids = implode(',', $filter['nodeIds'] ?: [0]);
		$user_ids = implode(',', $filter['userIds'] ?: [0]);
		$course_user_ids = implode(',', $filter['courseAttendeeIds'] ?: [0]);

		$filter['postTypes'] = array_map(fn($val) => Db_Helper::prepare('%s', [$val]), $filter['postTypes']);
		$post_types = implode(',', $filter['postTypes'] ?: ['']);


		$where = "WHERE p.post_type IN ({$post_types}) AND p.post_status = 'publish'";
		$where = Query_Helper::add_where($q->filter_query, $where, "AND", $q->filter_query);
		$where = Query_Helper::add_where($q->search_query, $where, "AND", $q->search_query);

		$category_node_where = ($filter['categoryIds'] || $filter['nodeIds']) ? "
			AND (category_id IN ({$category_ids}) OR category_id IS NULL)
			AND (node_id IN ({$node_ids}) OR node_id IS NULL)
		" : "";

		$category_node_where = ($filter['courseAttendeeIds']) ? "
			$category_node_where
			AND (c.user_id IN ({$course_user_ids}))
		" : $category_node_where;

		$filter_where = $where;

		if ($filter['userIds']) {
			$where = str_replace('WHERE', '', $where);
			$filter_where = "
				WHERE (
					{$where}
					AND u.user_id IN ({$user_ids})
				) OR (
					{$where}
					{$category_node_where}
				)
			";
		} else {
			$where = str_replace('WHERE', '', $where);
			$filter_where = "
				WHERE (
					{$where}
					{$category_node_where}
				)
			";
		}

		$where = $filter_where;

		$joins = "
			LEFT JOIN {$this->post_category_table} AS pc
			ON pc.post_id = p.ID
			LEFT JOIN {$this->post_node_table} AS pn
			ON pn.post_id = p.ID
			LEFT JOIN {$this->post_user_table} AS u
			ON u.post_id = p.ID
			LEFT JOIN {$this->course_user_table} AS c
			ON c.post_id = p.ID
		";

		$count_query = "SELECT COUNT(DISTINCT p.ID) AS num_rows FROM $this->post_table AS p $joins $where";

		$total_count = Db_Helper::get_row($count_query)->num_rows;
		Response_Helper::set_pagination_total_header($total_count);

		$posts = Db_Helper::get_rows("SELECT DISTINCT p.* FROM $this->post_table AS p $joins $where $q->order_by_query $q->limit_query");

		foreach ($posts as $post) {
			$meta = get_post_meta($post->ID);
			$post->meta = [];

			foreach ($meta as $key => $value) {
				$post->meta[$key] = $value[0];
			}

			$post->media_id = isset($post->meta['main_media_id']) ? intval($post->meta['main_media_id']) : null;
			$post->node_ids = $this->get_node_ids($post->ID);
			$post->category_ids = $this->get_category_ids($post->ID);
			$post->user_ids = $this->get_user_ids($post->ID);
			$post->editor_ids = $this->get_editor_ids($post->ID);
		}

		return $posts;
	}

	public function get_by_user_id(\WP_REST_Request $req, int $user_id): array
	{
		$q = new Query_Builder_Helper($req, Post_Query::$fields);

		$where = "WHERE p.post_type <> 'media'";

		$where = Query_Helper::add_where($q->filter_query, $where, "AND", $q->filter_query);
		$where = Query_Helper::add_where($q->search_query, $where, "AND", $q->search_query);

		$where = str_replace('WHERE', '', $where);
		$where = Db_Helper::prepare("
				WHERE (
					{$where}
					AND post_author = %d
				) OR (
					{$where}
					AND (
						SELECT COUNT(*) FROM {$this->post_editor_table} AS pe
					 	WHERE pe.post_id = p.ID AND pe.user_id = %d
						AND pe.type = 'editor'
					) = 1
				)
			", [$user_id, $user_id]);

		$count_query = "SELECT COUNT(DISTINCT p.ID) AS num_rows FROM $this->post_table AS p $where";
		$total_count = Db_Helper::get_row($count_query)->num_rows;
		Response_Helper::set_pagination_total_header($total_count);

		$posts = Db_Helper::get_rows("SELECT DISTINCT p.* FROM $this->post_table AS p $where $q->order_by_query $q->limit_query");

		foreach ($posts as $post) {
			$meta = get_post_meta($post->ID);
			$post->meta = [];

			foreach ($meta as $key => $value) {
				$post->meta[$key] = $value[0];
			}

			$post->media_id = isset($post->meta['main_media_id']) ? intval($post->meta['main_media_id']) : null;
			$post->node_ids = $this->get_node_ids($post->ID);
			$post->category_ids = $this->get_category_ids($post->ID);
			$post->user_ids = $this->get_user_ids($post->ID);
			$post->editor_ids = $this->get_editor_ids($post->ID);
		}

		return $posts;
	}

	public function create(object $body) : int {
		$entity = Post_Transformer::transform_in($body);

		$post_id = wp_insert_post($entity, false);

		if (!$post_id) {
			Response_Helper::die('Something went wrong creating the post');
		}

		$this->set_post_meta($post_id, $body);

		return $post_id;
	}

	private function set_post_meta(int $post_id, object $body)
	{
		$meta_fields = [
			'blurbs',
			'info',
			'contact',
			'unpublishedDate',
			'authoredDate',
			'pseudoAuthor'
		];

		foreach ($meta_fields as $field) {
			update_post_meta($post_id, $field, $body->{$field});
		}

		if ($body->course) {
			foreach ($body->course as $key => $value) {
				update_post_meta($post_id, "course_{$key}", $value);
			}
		}
	}

	public function update(int $post_id, object $body) : void
	{
		$entity = Post_Transformer::transform_in($body);

		$entity->ID = $post_id;
		wp_insert_post($entity, false);

		$this->set_post_meta($post_id, $body);
	}

	public function update_sort(array $posts) : void
	{
		foreach ($posts as $post) {
			Db_Helper::update('posts', array('menu_order' => $post->menu_order), array('id' => $post->ID));
		}
	}

	public function delete(int $id): void
	{
		Db_Helper::delete('posts', array('ID' => $id), ['%d']);
		Db_Helper::delete('post_category', array('post_id' => $id), ['%d']);
		Db_Helper::delete('post_node', array('post_id' => $id), ['%d']);
		Db_Helper::delete('post_user', array('post_id' => $id), ['%d']);
		Db_Helper::delete('post_editor', array('post_id' => $id), ['%d']);
		Db_Helper::delete('postmeta', array('post_id' => $id), ['%d']);
	}

	public function get_by_media_id(int $media_id) : array
	{
		return Db_Helper::get_rows("
			SELECT * FROM $this->post_table
			INNER JOIN $this->postmeta AS pm
			ON pm.meta_key = 'main_media_id'
			AND pm.meta_value = %d", [$media_id]);
	}

	public function get_sort_list(int $parent_id, int $sort, string $type, ?int $post_id) : array
	{
		$where = $post_id ? Db_Helper::prepare("AND p.id != %d", [$post_id]) : "";

		return Db_Helper::get_rows("
			SELECT * FROM {$this->post_table} AS p
			WHERE p.post_parent = %d
			AND p.menu_order > %d
			AND p.post_type = %s
			{$where}
			ORDER BY p.menu_order ASC
		", [$parent_id, $sort, $type]);
	}

	private function get_node_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_node_table WHERE post_id = %d", [$post_id]);

		return array_map(fn ($row): int => $row->node_id, $rows);
	}

	private function get_category_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_category_table WHERE post_id = %d", [$post_id]);

		return array_map(fn ($row): int => $row->category_id, $rows);
	}

	private function get_user_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_user_table WHERE post_id = %d", [$post_id]);

		return array_map(fn ($row): int => $row->user_id, $rows);
	}

	private function get_editor_ids(int $post_id): array
	{
		$rows = Db_Helper::get_rows("SELECT * FROM $this->post_editor_table WHERE post_id = %d AND type = 'editor'", [$post_id]);

		return array_map(fn ($row): int => $row->user_id, $rows);
	}
}
