<?php

namespace Rubicore\Core;

use \WP_REST_Request, \WP_User;

final class Course_Controller
{

	function __construct()
	{
		$this->routes();
	}

	private function routes(): void
	{
		$api = new Api_Helper('rubicore/core/v1');

		$api->add_route('course', 'GET', function (WP_REST_Request $req, WP_User $user) : array {
				User_Helper::allowed_to('course_list', $user);
				$post_service = new Post_Service();

				return $post_service->get($req, $user, ['course']);
			},
			array(
				'args' => Post_Service::get_args()
			)
		);

		$api->add_route('course/(?P<slug>[^/]+?)', 'GET', function (WP_REST_Request $req, WP_User $user): Post_Model {
			$post = Post_Service::get_by_slug($req->get_param('slug'), 'course');

			if (!$post) {
				Response_Helper::not_found('No post found with provided id.');
			}

			User_Helper::allowed_to('post_get', $user, [$post->author->id, ...$post->editorIds]);

			return $post;
		}, array(
			'args' => array(
				'slug' => array(
					'type' => [Api_Helper::TYPE_INT, Api_Helper::TYPE_STRING]
				)
			)
		));


		$api->add_route('course', 'POST', function (WP_REST_Request $req, WP_User $user) : int {
				User_Helper::allowed_to('course_add', $user);

				return Post_Service::post($req, $user);
			},
			array(
				'args' => array_merge(
					Post_Service::post_args(),
					Course_Service::post_args()
				)
			)
		);


		$api->add_route('course/(?P<post_id>[^/]+?)/attendee/(?P<user_id>[^/]+?)', 'POST', function (WP_REST_Request $req, WP_User $user): void {
			$post_id = $req->get_param('post_id');
			$user_id = $req->get_param('user_id');
			$body = Request_Helper::get_body($req);

			Course_Service::add_attendee($post_id, $user_id, $body, $user);
		}, [
			'args' => Course_Service::attendee_post_args()
		]);


		$api->add_route('course/(?P<post_id>[^/]+?)/attendee/(?P<user_id>[^/]+?)', 'PUT', function (WP_REST_Request $req, WP_User $user) : void {
			$post_id = $req->get_param('post_id');
			$user_id = $req->get_param('user_id');
			$body = Request_Helper::get_body($req);

			Course_Service::update_attendee($post_id, $user_id, $body, $user);
		}, [
			'args' => Course_Service::attendee_put_args()
		]);

		$api->add_route('course/(?P<post_id>[^/]+?)/attendee/(?P<user_id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user): void {
			$post_id = $req->get_param('post_id');
			$user_id = $req->get_param('user_id');

			Course_Service::remove_attendee($post_id, $user_id, $user);
		}, [
			'args' => [
				'post_id' => [
					'in' => Api_Helper::IN_PATH,
					'type' => Api_Helper::TYPE_INT,
					'required' => true,
					'minimum' => 1
				],

				'user_id' => [
					'in' => Api_Helper::IN_PATH,
					'type' => Api_Helper::TYPE_INT,
					'required' => true,
					'minimum' => 1
				]
			]
		]);

	}
}
