<?php

namespace Rubicore\Core;

final class Token_Helper {

	public static function get_from_request(\WP_REST_Request $req) : ?string {
		$header = $req->get_header('Authorization');

		if (is_null($header)) {
			return null;
		}

		return substr($header, 7);
	}

	public static function get_expire_time(string $type) : int {
		switch ($type) {
			case 'web':
				return time() + RUBICORE_TOKEN_TIME_WEB;
			case 'api':
				return time() + RUBICORE_TOKEN_TIME_API;
		}
	}

	public static function create($payload) {
		return array(
			'token' => self::jwt($payload),
			'expires' => self::get_expire_time($payload->type)
		);
	}

	public static function jwt($content, $header = '') {
		$secret = RUBICORE_SECRET;
		$payload = Format_Helper::b64url(json_encode($content));

		if ($header == '') {
			$header = Format_Helper::b64url(json_encode([
				'typ' => 'JWT',
				'alg' => 'HS256',
				'exp' => self::get_expire_time($content->type)
			]));
		}

		$signature = Format_Helper::b64url(hash_hmac('sha256', $header . '.' . $payload, $secret, true));
		$jwt = $header . '.' . $payload . '.' . $signature;

		return $jwt;
	}


	private static function split($token) {
		$array = explode('.', $token);

		if (count($array) != 3) {
			return null;
		}

		return array(
			'header' => Format_Helper::b64urldecode($array[0]),
			'payload' => Format_Helper::b64urldecode($array[1]),
			'signature' => $array[2]
		);
	}

	public static function validate($token) {
		$t = self::split($token);

		if ($t == null) {
			return array('is_valid' => false, 'msg' => 'Invalid token format');
		}

		$payload = json_decode($t['payload']);
		$testToken = self::jwt($payload, Format_Helper::b64url($t['header']));

		if ($testToken != $token) {
			return array('is_valid' => false, 'msg' => 'Token mismatch');
		}

		$allowedClockDrift = 60; // seconds
		$header = json_decode($t['header']);

		if ($header->exp < (time() - $allowedClockDrift)) {
			return array('is_valid' => false, 'msg' => 'Token expired');
		}

		return array('is_valid' => true, 'msg' => 'Token is valid');
	}

	public static function get_user_id(string $token) : ?int {
		$t = self::split($token);

		return $t ? intval(json_decode($t['payload'])->userId) : null;
	}

}
