<?php

namespace Rubicore\Core;
use \WP_REST_Request, \WP_User;

final class Media_Controller {

	function __construct () {
		$this->routes();
	}


	private function routes() : void {
		$api = new Api_Helper('rubicore/v1');

		$api->add_route('media', 'GET', function (WP_REST_Request $req, WP_User $user) : array {
			User_Helper::allowed_to('media_list', $user);

			$media_repo = new Media_Repository();
			$user_repo = new User_Repository();
			$filters = array(
				'nodeIds' => $req->get_param('nodeId')
			);

			$medias = $media_repo->get($req, $filters);

			return array_map(function($media) use($user_repo) : Media_Model {
				$media->user = $user_repo->get_by_id($media->post_author);

				return Media_Transformer::transform_out($media);
			}, $medias);
		}, array(
			'args' => array_merge(
				Media_Query::get_args(),
				Query_Helper::get_pagination_args(),
				array(
					'nodeId' => array(
						'type' => 'array',
						'required' => false,
						'uniqueItems' => true,
						'items' => array(
							'type' => 'integer',
							'minimum' => 1
						)
					)
				)
			)
		));

		$api->add_route('media/(?P<media_id>[^/]+?)', 'GET', function (WP_REST_Request $req, WP_User $user) : Media_Model {
			$media_repo = new Media_Repository();
			$media = $media_repo->get_by_id($req->get_param('media_id'));

			if (!$media) {
				Response_Helper::not_found('No media found with provided id.');
			}

			$user_repo = new User_Repository();
			$media->user = $user_repo->get_by_id($media->post_author);

			User_Helper::allowed_to('media_get', $user, $media->post_author);

			return Media_Transformer::transform_out($media);
		}, array(
			'args' => array(
				'media_id' => array('type' => 'integer', 'required' => true, 'minimum' => 1)
			)
		));


		$api->add_route('media', 'POST', function (WP_REST_Request $req, WP_User $user) : int {
			User_Helper::allowed_to('media_create', $user);

			$media_repo = new Media_Repository();
			$allowed_types = ['jpeg', 'jpg', 'png', 'webp', 'gif'];
			$file = $req->get_body();
			$content_length = $req->get_header('content_length');
			$content_type = $req->get_content_type();

			if (!$content_length) {
				Response_Helper::die('Missing request header "content-length"');
			}

			if (!$content_type) {
				Response_Helper::die('Missing request header "content-type"');
			}

			if (!in_array($content_type['subtype'], $allowed_types)) {
				Response_Helper::die('Unsupported file-type ' . $content_type['subtype']);
			}

			$mbs = Format_Helper::bytes_to_mb($content_length);

			if ($mbs >= 50) {
				Response_Helper::die('Exceeding maximum allowed file-size of 50MB: ' . $mbs . 'MB');
			}

			$post_data = new \stdClass();
			$post_data->title = $req->get_param('title') ?: '';
			$post_data->description = $req->get_param('description') ?: '';
			$post_data->userId = $user->ID;

			return $media_repo->create($file, $content_type, $post_data);
		}, array(
			'args' => array(
				'title' => array(
					'type' => 'string',
					'required' => true,
					'minLength' => 1
				),
				'description' => array(
					'type' => 'string',
					'required' => false,
					'minLength' => 1
				)
			)
		));


		$api->add_route('media/(?P<media_id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user) : void {
			$post_repo = new Post_Repository();
			$media_post = $post_repo->get_by_slug($req->get_param('media_id'), null);

			if (!$media_post) {
				Response_Helper::not_found('No media found with provided id.');
			}

			$post = $post_repo->get_by_media_id($req->get_param('media_id'));

			if (count($post) != 0) {
				Response_Helper::die('Unable to delete media connected to post.');
			}

			User_Helper::allowed_to('media_delete', $user, $media_post->post_author);

			$media_repo = new Media_Repository();
			$media_repo->delete($req->get_param('media_id'), $media_post->post_content);
		}, array(
			'args' => array(
				'media_id' => array('type' => 'integer', 'required' => true, 'minimum' => 1)
			)
		));

	}
}
