<?php

namespace Rubicore\Core;
use \WP_REST_Request, \WP_User;

final class Link_Controller {

	function __construct () {
		$this->routes();
	}


	private function routes() : void {
		$api = new Api_Helper('rubicore/v1');

		$api->add_route('link', 'GET', function (WP_REST_Request $req, WP_User $user) : array {
			User_Helper::allowed_to('get_links', $user);

			$link_repo = new Link_Repository();
			$user_repo = new User_Repository();
			$links = $link_repo->get($req, $req->get_param('userId'));

			return array_map(function($link) use($user_repo) : Link_Model {
				$link->user = $user_repo->get_by_id($link->author);
				return Link_Transformer::transform_out($link);
			}, $links);
		}, array(
			'args' => array_merge(
				array(
					'userId' => array('type' => 'integer', 'required' => false, 'minimum' => 1)
				),
				Link_Query::get_args(),
				Query_Helper::get_pagination_args()
			)
		));



		$api->add_route('link/(?P<link_id>[^/]+?)', 'GET', function (WP_REST_Request $req, WP_User $user) : Link_Model {
			$link_repo = new Link_Repository();
			$link = $link_repo->get_by_id($req->get_param('link_id'));

			if (!$link) {
				Response_Helper::not_found('No link found with provided id.');
			}

			User_Helper::allowed_to('get_links', $user, intval($link->author));

			$nodes = $link_repo->get_nodes([$link]);
			$user_repo = new User_Repository();
			$link->user = $user_repo->get_by_id($link->author);

			return Link_Transformer::transform_out($link, $nodes);
		}, array(
			'args' => array(
				'link_id' => array('type' => 'integer', 'required' => false, 'minimum' => 1)
			)
		));



		$api->add_route('link/(?P<link_id>[^/]+?)', 'DELETE', function (WP_REST_Request $req, WP_User $user) : void {
			$link_repo = new Link_Repository();
			$link_id = $req->get_param('link_id');
			$link = $link_repo->get_by_id($link_id);

			if (!$link) {
				Response_Helper::not_found('No link found with provided id.');
			}

			User_Helper::allowed_to('delete_links', $user, intval($link->author));

			$link_repo->delete_link($link_id);
		}, array(
			'args' => array(
				'link_id' => array('type' => 'integer', 'required' => false, 'minimum' => 1)
			)
		));



		$api->add_route('link/(?P<link_id>[^/]+?)', 'PUT', function (WP_REST_Request $req, WP_User $user) : void {
			$link_repo = new Link_Repository();
			$link_id = $req->get_param('link_id');
			$body = Request_Helper::get_body($req);

			$link = $link_repo->get_by_id($link_id);
			$author = $body->author ? $body->author : $user->ID;

			if (!$link) {
				Response_Helper::not_found('No link found with provided id.');
			}

			User_Helper::allowed_to('update_links', $user, intval($link->author));

			$link_repo->update_link($link_id, Link_Transformer::transform_in($body, $author));
		}, array(
			'args' => array(
				'link_id' => array('type' => 'integer', 'required' => false, 'minimum' => 1),
				'name' => array('type' => 'string', 'minLength' => 2, 'required' => true),
				'type' => array('type' => 'string', 'minLength' => 3, 'required' => true),
				'url' => array('type' => 'string', 'minLength' => 10, 'required' => true),
				'categoryItemIds' => array('type' => 'array', 'required' => false, 'items' => array(
					'type' => 'integer', 'required' => true, 'minimum' => 1
				)),
				'shortDescription' => array('type' => 'string', 'required' => false),
				'longDescription' => array('type' => 'string', 'required' => false),
				'icon' => array('type' => 'string', 'required' => false),
				'author' => array('type' => 'integer', 'required' => false, 'minimum' => 1),
				'public' => array('type' => 'boolean', 'required' => false, 'default' => true)
			)
		));



		$api->add_route('link', 'POST', function (WP_REST_Request $req, WP_User $user) : int {
			User_Helper::allowed_to('post_links', $user);

			$link_repo = new Link_Repository();
			$user_repo = new User_Repository();
			$body = Request_Helper::get_body($req);

			$link_entity = Link_Transformer::transform_in($body, $user->ID);
			$link_id = $link_repo->create($link_entity);

			if ($body->nodeIds) {

				foreach ($body->nodeIds as $node_id) {
					$link_repo->add_link_to_node(array(
						'node_id' => $node_id,
						'link_id' => $link_id
					));


					if ($link_entity->public) {
						// TODO: Performance, this logic is heavy if alot of user is in the selection
						$users = $user_repo->get_users_by_node($node_id);

						foreach ($users as $user_id) {
							$link_repo->add_link_to_user(array(
								'user_id' => $user_id,
								'link_id' => $link_id
							));
						}
					}

				}
			}

			return $link_id;
		}, array(
			'args' => array(
				'name' => array('type' => 'string', 'minLength' => 2, 'required' => true),
				'type' => array('type' => 'string', 'minLength' => 3, 'required' => true),
				'url' => array('type' => 'string', 'minLength' => 10, 'required' => true),
				'nodeIds' => array('type' => 'array', 'required' => false, 'items' => array(
					'type' => 'integer', 'required' => true, 'minimum' => 1
				)),
				'shortDescription' => array('type' => 'string', 'required' => false),
				'longDescription' => array('type' => 'string', 'required' => false),
				'icon' => array('type' => 'string', 'required' => false),
				'author' => array('type' => 'integer', 'required' => false, 'minimum' => 1),
				'public' => array('type' => 'boolean', 'required' => false, 'default' => true)
			)
		));

	}
}
