<?php

namespace Rubicore\Core;

use \WP_REST_Request, \WP_User;

final class Like_Controller
{

	function __construct()
	{
		$this->routes();
	}

	private function routes(): void
	{
		$api = new Api_Helper('rubicore/v1');

		$api->add_route('like', 'GET', function (WP_REST_Request $req, WP_User $user): array {
			User_Helper::allowed_to('like_list', $user);
			$like_repo = new Like_Repository();
			$user_repo = new User_Repository();
			$likes = $like_repo->get($req);

			return array_map(function ($like) use ($user_repo): Like_Model {
				$like->user = $user_repo->get_by_id($like->user_id);
				return Like_Transformer::transform_out($like);
			}, $likes);
		}, array(
			'args' => array_merge(
				Like_Query::get_args(),
				Query_Helper::get_pagination_args()
			)
		));


		$api->add_route('like', 'POST', function (WP_REST_Request $req, WP_User $user): int {
			User_Helper::allowed_to('like_create', $user);

			$body = Request_Helper::get_body($req);
			$post_id = $body->postId;
			$type = $body->type;

			$post = (new Post_Repository())->get_by_slug($post_id, null);
			if (!$post) {
				Response_Helper::not_found("No post found with provided id.");
			}

			if ($post->comment_status !== 'open') {
				Response_Helper::die('Likes disabled for post with provided id.');
			}

			$like_repo = new Like_Repository();

			return $like_repo->create(Like_Transformer::transform_in(array(
				'post_id' => $post_id,
				'user_id' => $user->ID,
				'type' => $type
			)));
		}, array(
			'args' => array(
				'postId' => array('type' => 'integer', 'required' => true, 'minimum' => 1),
				'type' => array('type' => 'string', 'required' => true, 'minLength' => 1)
			)
		));

		$api->add_route('like/(?P<like_id>\d+)', 'DELETE', function (WP_REST_Request $req, WP_User $user): void {
			$like_repo = new Like_Repository();
			$like = $like_repo->get_by_id($req->get_param('like_id'));

			if (!$like) {
				Response_Helper::not_found("No like found with provided id.");
			}

			User_Helper::allowed_to('like_delete', $user, $like->user_id);

			$like_repo->delete($req->get_param('like_id'));
		}, array(
			'args' => array(
				'like_id' => array('type' => 'integer', 'required' => true, 'minimum' => 1)
			)
		));

		$api->add_route('like/(?P<like_id>\d+)', 'PUT', function (WP_REST_Request $req, WP_User $user): void {
			$like_repo = new Like_Repository();
			$like = $like_repo->get_by_id($req->get_param('like_id'));
			$body = Request_Helper::get_body($req);

			if (!$like) {
				Response_Helper::not_found("No like found with provided id.");
			}

			User_Helper::allowed_to('like_edit', $user, $like->user_id);

			$like_repo->update($req->get_param('like_id'), $body->type);
		}, array(
			'args' => array(
				'like_id' => array('type' => 'integer', 'required' => true, 'minimum' => 1),
				'type' => array('type' => 'string', 'required' => true, 'minLength' => 1)
			)
		));
	}
}
