<?php

namespace Rubicore\Alfresco;

use \Rubicore\Core\Response_Helper;

final class Request_Helper
{
	private static string $base_url = "https://alfresco.alingsas.se/alfresco/api/-default-/public/alfresco/versions/1/";

	private static function get_url(string $path, array $options, string $base_url = '') : string
	{
		$final_url = $base_url ?
			$base_url . $path :
			self::$base_url . $path;

		$final_url = array_key_exists('query', $options) ?
			add_query_arg($options['query'], $final_url) :
			$final_url;

		return $final_url;
	}

	public static function post(string $path, array $options, string $base_url = '')
	{
		return self::call('POST', self::get_url($path, $options, $base_url), $options);
	}

	public static function get(string $path, array $options, string $base_url = '')
	{
		return self::call('GET', self::get_url($path, $options, $base_url), $options);
	}

	public static function delete(string $path, array $options, string $base_url = '')
	{
		return self::call('DELETE', self::get_url($path, $options, $base_url), $options);
	}

	private static function call(string $method, string $url, array $options)
	{
		$body = array_key_exists('body', $options) ? $options['body'] : [];
		$headers = ['Content-Type' => 'application/json'];

		if (array_key_exists('token', $options)) {
			if (strlen($options['token']) == 0) {
				Response_Helper::not_authenticated('Bad token format');
			}

			$headers['Authorization'] = "Basic " . base64_encode($options['token']);
		}

		if (array_key_exists('headers', $options)) {
			foreach($options['headers'] as $key => $value) {
				$headers[$key] = $value;
			}
		}

		$remote_options = [
			'method' => $method,
			'timeout' => 45,
			'headers' => $headers,
			'body' => wp_json_encode($body)
		];

		$response = wp_remote_request($url, $remote_options);

		if (is_wp_error($response)) {
			$error_message = $response->get_error_message();
			Response_Helper::die($error_message);
		}

		$body = json_decode($response['body'], true);

		if (array_key_exists('error', $body)) {
			Response_Helper::die($body['error']['briefSummary'], $body['error']['statusCode']);
		}

		return $body;
	}

}
