<?php

namespace EndorfinKnowe;

use \add_action, \add_shortcode;

final class Dynamic_Supply_Shortcode
{

    function __construct()
    {
        $this->init();
    }

    private function init()
    {
        add_action('init', function () {
            $this->add_style();
			$this->add_script_file();
            $this->register_shortcode();
		});
    }

    private function add_style() : void
	{
		wp_enqueue_style('dynamic-supply-shortcode-styles', plugin_dir_url(__FILE__) . 'styles/index.css', array(), '1.0.0');
	}

    private function add_script_file()
    {
        wp_enqueue_script("dynamic-supply-shortcode-scripts", plugin_dir_url(__FILE__) . "scripts/index.js", array(), "1.0.0", true);

        add_filter('script_loader_tag', function ($tag, $handle, $src) {
			// if not your script, do nothing and return original $tag
			if ('dynamic-supply-shortcode-scripts' !== $handle) {
				return $tag;
			}

			// change the script tag by adding type="module" and return it.
			$tag = '<script type="module" src="' . esc_url( $src ) . '"></script>';

			return $tag;
		} , 10, 3);
    }

    private function register_shortcode() : void
    {
        add_shortcode( "supply-accordion", fn($attr) => $this->shortcode_callback($attr) );
    }

    private function shortcode_callback($attributes) : string
    {
        $supplies = $this->get_supplies();

        $category_id = $this->get_category_id_from_supplies($supplies, $attributes["category-slug"]);
        $facility_id = $this->get_facility_id_from_supplies($supplies, $attributes["facility-slug"]);

        if($category_id == null || $facility_id == null)
            return "";

        $supplies_exist = $this->supplies_exist($supplies, $category_id, $facility_id);
        if(!$supplies_exist)
            return "";

        $category_supplies = $this->get_category_supplies($supplies, $attributes["category-slug"]);
        $accordion_id = $this->build_accordion_id($category_supplies, $attributes["category-slug"]);
        $supplies_json = json_encode($supplies);

        $html = "

			<div 
                id='{$accordion_id}' 
                class='dynamic-supply-accordion' 
                data-facilityId='{$facility_id}' 
                data-facility='{$attributes["facility-slug"]}' 
                data-category='{$attributes["category-slug"]}' 
                data-categoryId='{$category_id}' 
                data-title='{$attributes["title"]}'>
			</div>

			<script>
				window._SUPPLIES_DATA = {$supplies_json};
			</script>

		";

        return $html;
    }

    private function get_category_id_from_supplies($supplies, $category_slug) {
        $category_supplies = $this->get_category_supplies($supplies, $category_slug);
        $category_id = $this->get_category_id($category_supplies, $category_slug);

        return $category_id;
    }

    private function get_facility_id_from_supplies($supplies, $facility_slug) {
        $facility_supplies = $this->get_facility_supplies($supplies, $facility_slug);
        $facility_id = $this->get_facility_id($facility_supplies, $facility_slug);

        return $facility_id;
    }

    private function get_facility_id($supplies, $facility_slug) {
        $facility = $this->get_facility_by_slug($supplies, $facility_slug);
        $facility_id = $facility->term_id;

        return $facility_id;
    }

    private function get_facility_by_slug($supplies, $sought_slug) {
        $facilities = $this->get_facilities_from_supplies($supplies);
        
        foreach($facilities as $facility) {
            $sought_slug = strtoupper($sought_slug);
            $facility_slug = strtoupper($facility->slug);
            
            if($sought_slug == $facility_slug)
                return $facility;
        }

        return null;
    }

    private function get_facility_supplies($supplies, $facility) {
        $facility_supplies = array();
        foreach($supplies as $supply) 
            if($this->supply_is_in_facility($supply, $facility))
                $facility_supplies[] = $supply;

        return $facility_supplies;
    }

    private function build_accordion_id($category_supplies, $category_slug) {
        $category_id = $this->get_category_id($category_supplies, $category_slug);
        $hash = $this->build_hash();
        $accordion_id = "dynamic-supply-accordion-{$category_id}-{$hash}";

        return $accordion_id;
    }

    private function build_hash() {
        $random = rand();
        $hash = hash('sha256', $random . time());

        return $hash;
    }

    private function get_supplies() : Array {
        $args = [
			'post_type' => 'supplies',
			'posts_per_page' => 9999
		];

		$query = new \WP_Query($args);
		$rows = $query->posts;

		$supplies = array_map(fn($row) => [
			'id' => $row->ID,
			'terms' => get_the_terms($row->ID, 'supplies_type'),
			'facilities' => get_the_terms($row->ID, 'employee_facility'),
			'time' => get_post_meta($row->ID, 'Time', true),
			'condition' => get_post_meta($row->ID, 'Kondition', true),
			'strength' => get_post_meta($row->ID, 'Styrka', true),
			'balance' => get_post_meta($row->ID, 'Balans', true),
			'choreography' => get_post_meta($row->ID, 'Koreografi', true),
			'intensity' => get_post_meta($row->ID, 'Intensitet', true),
			'mobility' => get_post_meta($row->ID, 'Rörlighet', true),
			'custom_buttons'  => get_post_meta($row->ID, 'custom_buttons', true),
			'title' => $row->post_title,
			'excerpt' => $row->post_excerpt,
			'image' => get_the_post_thumbnail_url($row)
		], $rows);

        return $supplies;
    }

    private function get_category_supplies($supplies, $category_slug) : array
    {
        $category_supplies = array();
        foreach($supplies as $supply) 
            if($this->supply_is_in_category($supply, $category_slug))
                $category_supplies[] = $supply;

        return $category_supplies;
    }

    private function supply_is_in_category($supply, $sought_slug) {
        $supply_categories = $supply["terms"];

        foreach($supply_categories as $supply_category) {
            $sought_slug = strtoupper(trim($sought_slug));
            $category_slug = strtoupper(trim($supply_category->slug));

            if($category_slug == $sought_slug)
                return true;
        }

        return false;
    }

    private function supply_is_in_facility($supply, $sought_slug) : bool {
        $supply_facilities = $supply["facilities"];

        foreach($supply_facilities as $supply_facility) {
            $sought_slug = strtoupper(trim($sought_slug));
            $facility = strtoupper(trim($supply_facility->slug));

            if($facility == $sought_slug)
                return true;
        }

        return false;
    }

    private function get_category_id($category_supplies, $category_slug) {
        $category = $this->get_category_by_slug($category_supplies, $category_slug);
        $category_id = $category->term_id;

        return $category_id;
    }

    private function get_category_by_slug($supplies, $sought_slug) {
        $categories = $this->get_categories_from_supplies($supplies);

        $found_category = null;
        foreach($categories as $category) {
            $sought_slug = strtoupper($sought_slug);
            $category_slug = strtoupper($category->slug);

            if($sought_slug == $category_slug) {
                $found_category = $category;
                break;
            }
        }

        return $found_category;
    }

    private function get_categories_from_supplies($supplies) {
        $categories = [];

        foreach($supplies as $supply) {
            $terms = $supply["terms"];
            $categories = $this->add_items_to_array($categories, $terms);
        }

        return $categories;
    }

    private function add_items_to_array($array, $added_items) {
        foreach($added_items as $added_item) {
            $array[] = $added_item;
        }

        return $array;
    }

    private function get_facilities_from_supplies($supplies) {
        $facilities = [];

        foreach($supplies as $supply) {
            $facilities = $supply["facilities"];
            $facilities = $this->add_items_to_array($facilities, $facilities);
        }

        return $facilities;
    }

    private function supplies_exist($supplies, $category_id, $facility_id) : bool {
        foreach($supplies as $supply) {
            $supply_exists = $this->supply_exists_in_category_and_facility($supply, $category_id, $facility_id);

            if($supply_exists)
                return true;
        }

        return false;
    }

    private function supply_exists_in_category_and_facility($supply, $category_id, $facility_id) : bool {
        $supply_array = [$supply];

        $facilities = $this->get_facilities_from_supplies($supply_array);
        $facility = $this->get_facility_by_id($facilities, $facility_id);
        if($facility == null)
            return false;

        $categories = $this->get_categories_from_supplies($supply_array);
        $category = $this->get_category_by_id($categories, $category_id);
        if($category == null)
            return false;

        return true;
    }

    private function get_facility_by_id($facilities, $sought_id) {
        $found_facility = null;

        foreach($facilities as $facility) {
            $facility_id = $facility->term_id;

            if($sought_id == $facility_id) {
                $found_facility = $facility;
                break;
            }
        }

        return $found_facility;
    }

    private function get_category_by_id($categories, $sought_id) {
        $found_category = null;

        foreach($categories as $category) {
            $category_id = $category->term_id;

            if($sought_id == $category_id) {
                $found_category = $category;
                break;
            }
        }

        return $found_category;
    }

}