<?php

namespace EndorfinKnowe;

use \add_action, \add_shortcode;

final class Dynamic_Supply_Shortcode
{

    function __construct()
    {
        $this->init();
    }

    private function init()
    {
        add_action('init', function () {
			$this->add_script_file();
            $this->register_shortcode();
		});
    }

    private function add_script_file()
    {
        wp_enqueue_script("dynamic-supply-shortcode-scripts", plugin_dir_url(__FILE__) . "scripts/index.js", array(), "1.0.0", true);

        add_filter('script_loader_tag', function ($tag, $handle, $src) {
			// if not your script, do nothing and return original $tag
			if ('dynamic-supply-shortcode-scripts' !== $handle) {
				return $tag;
			}

			// change the script tag by adding type="module" and return it.
			$tag = '<script type="module" src="' . esc_url( $src ) . '"></script>';

			return $tag;
		} , 10, 3);
    }

    private function register_shortcode() : void
    {
        add_shortcode( "supply-accordion", fn($attr) => $this->shortcode_callback($attr) );
    }

    private function shortcode_callback($attributes) : string
    {
        $attributes = $this->read_shortcode_attributes($attributes);

        $category_title = $attributes["title"];

        $supplies = $this->get_supplies();
        $supplies_json = json_encode($supplies);

        $category_supplies = $this->get_category_supplies($supplies, $category_title);

        if(empty($category_supplies))
            return "";
        
        $accordion_id = $this->build_accordion_id($category_supplies);

        $html = "

			<div id='{$accordion_id}' class='dynamic-supply-accordion'>
			</div>

			<script>
				window._SUPPLIES_DATA = {$supplies_json};
			</script>

		";

        return $html;
    }

    private function build_accordion_id($category_supplies) {
        $category_id = $this->get_category_id($category_supplies);
        $hash = $this->build_hash();
        $accordion_id = "dynamic-supply-accordion-{$category_id}-{$hash}";

        return $accordion_id;
    }

    private function build_hash() {
        $random = rand();
        $hash = hash('sha256', $random . time());

        return $hash;
    }

    private function read_shortcode_attributes($attributes) {
        $attributes = shortcode_atts( array(
            'title' => ''
         ), $attributes );

         return $attributes;
    }

    private function get_supplies() : Array {
        $args = [
			'post_type' => 'supplies',
			'posts_per_page' => 9999
		];

		$query = new \WP_Query($args);
		$rows = $query->posts;

		$supplies = array_map(fn($row) => [
			'id' => $row->ID,
			'terms' => get_the_terms($row->ID, 'supplies_type'),
			'time' => get_post_meta($row->ID, 'Time', true),
			'condition' => get_post_meta($row->ID, 'Kondition', true),
			'strength' => get_post_meta($row->ID, 'Styrka', true),
			'balance' => get_post_meta($row->ID, 'Balans', true),
			'choreography' => get_post_meta($row->ID, 'Koreografi', true),
			'custom_buttons'  => get_post_meta($row->ID, 'custom_buttons', true),
			'title' => $row->post_title,
			'excerpt' => $row->post_excerpt,
			'image' => get_the_post_thumbnail_url($row)
		], $rows);

        return $supplies;
    }

    private function get_category_supplies($supplies, $category_title) : array
    {
        $category_supplies = array();
        foreach($supplies as $supply) 
            if($this->supply_is_in_category($supply, $category_title))
                $category_supplies[] = $supply;

        return $category_supplies;
    }

    private function supply_is_in_category($supply, $sought_title) {
        $supply_categories = $supply["terms"];

        foreach($supply_categories as $supply_category) {
            $sought_title = strtoupper(trim($sought_title));
            $category_title = strtoupper(trim($supply_category->name));

            if($category_title == $sought_title)
                return true;
        }

        return false;
    }

    private function get_category_id($category_supplies) {
        $supply = $category_supplies[0];
        $supply_terms = $supply["terms"];
        $category_id = $supply_terms[0]->term_id;

        return $category_id;
    }
}